<?php

declare(strict_types=1);

namespace App\Controller;

use App\Entity\Product;
use App\Form\Type\AdminProductExportType;
use App\Form\Type\AdminProductFinalizeImportType;
use App\Form\Type\AdminProductImportType;
use App\Form\Type\AdminProductReportingType;
use App\Service\ProductService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class AdminProductController extends AbstractController
{
    /**
     * @param Request                $request
     * @param EntityManagerInterface $entityManager
     *
     * @return Response
     */
    public function finalizeImport(Request $request, EntityManagerInterface $entityManager): Response
    {
        $form = $this->createForm(AdminProductFinalizeImportType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $data = $form->getData();
            $result = $entityManager->getRepository(Product::class)->finalizeImport($data['taxon']);

            if (true === $result) {
                $this->addFlash('success', 'Import terminé avec succès.');
            } else {
                $this->addFlash('error', $result);
            }
        }

        return $this->render('admin/product/finalizeImport.html.twig', [
            'form' => $form->createView(),
        ]);
    }

    /**
     * @param Request        $request
     * @param ProductService $productService
     *
     * @return Response
     */
    public function reporting(Request $request, ProductService $productService): Response
    {
        $error = false;
        $form = $this->createForm(AdminProductReportingType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $parameters = $productService->reporting($form->getData()['file']);

            if (array_key_exists('error', $parameters)) {
                $error = $parameters['error'];
            } else {
                $form = $this->createForm(AdminProductExportType::class, [
                    'file' => $parameters['fileName'],
                ], [
                    'action' => $this->generateUrl('app_admin_products_export'),
                ]);

                return $this->render('admin/product/reportingResult.html.twig', $parameters + ['form' => $form->createView()]);
            }
        }

        return $this->render('admin/product/reporting.html.twig', [
            'form' => $form->createView(),
            'error' => $error,
        ]);
    }

    /**
     * @param Request        $request
     * @param ProductService $productService
     *
     * @return Response
     *
     * @throws \Exception
     */
    public function export(Request $request, ProductService $productService): Response
    {
        $form = $this->createForm(AdminProductExportType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            return $this->file($productService->export($form->getData()['file']));
        }

        throw $this->createNotFoundException();
    }

    /**
     * @param Request        $request
     * @param ProductService $productService
     *
     * @return Response
     *
     * @throws \Exception
     */
    public function import(Request $request, ProductService $productService): Response
    {
        $form = $this->createForm(AdminProductImportType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            try {
                $productService->import($form->getData()['file']);

                $this->addFlash('success', "L'import a été effectué.");
            } catch (\Exception $e) {
                $this->addFlash('error', $e->getMessage());
            }
        }

        return $this->render('admin/product/import.html.twig', [
            'form' => $form->createView(),
        ]);
    }
}
