<?php

declare(strict_types=1);

namespace App\Controller;

use App\Entity\Product;
use Doctrine\ORM\EntityManagerInterface;
use Sylius\Component\Core\Model\ChannelPricing;
use Sylius\Component\Core\Model\ProductVariant;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;
use Symfony\Component\Serializer\SerializerInterface;

class ApiProductController extends AbstractController
{
    public function search(Request $request): Response
    {
        $reference = $request->query->get('reference');

        if (!$reference) {
            return $this->json([]);
        }

        $products = $this->getDoctrine()
            ->getRepository(Product::class)
            ->searchByReference($reference);
        $result = [];

        /** @var Product $product */
        foreach ($products as $product) {
            $result[] = [
                'id' => $product->getCode(),
                'affichage' => $product->getManufacturerReference(),
            ];
        }

        return $this->json($result);
    }

    public function find(Request $request): Response
    {
        $reference = $request->query->get('reference');

        if (!$reference) {
            throw $this->createNotFoundException();
        }

        /** @var Product $product */
        $product = $this->getDoctrine()
            ->getRepository(Product::class)
            ->findOneBy(['code' => $reference]);

        if (!$product) {
            throw $this->createNotFoundException();
        }

        /** @var ProductVariant $productVariant */
        $productVariant = $product->getVariants()->first();

        /** @var ChannelPricing $channelPricing */
        $channelPricing = $productVariant->getChannelPricings()->first();

        return $this->json([
            'code' => $product->getCode(),
            'manufacturer_reference' => $product->getManufacturerReference(),
            'designation' => $product->getShortDescription(),
            'price_without_vat' => $channelPricing->getPrice(),
            'weight' => $productVariant->getWeight(),
        ]);
    }

    public function findSearchableProductByTaxon(Request $request, EntityManagerInterface $entityManager, SerializerInterface $serializer): JsonResponse
    {
        if ($request->isXmlHttpRequest()) {
            $taxonId = $request->query->getInt('id_taxon');

            if (empty($taxonId)) {
                return new JsonResponse([]);
            }

            $products = $entityManager->getRepository(Product::class)->findSearchableProductByTaxon($taxonId);

            foreach ($products as $product) {
                $product->setUrl(
                    $this->generateUrl('sylius_shop_product_show', [
                        'slug' => $product->getSlug(), ],
                        UrlGeneratorInterface::ABSOLUTE_URL
                    )
                );
            }

            $jsonProducts = $serializer->serialize($products, 'json', ['attributes' => ['id', 'name', 'url']]);

            return new JsonResponse($jsonProducts, JsonResponse::HTTP_OK, [], true);
        }

        throw new AccessDeniedException();
    }
}
