<?php

declare(strict_types=1);

namespace App\DataFixtures\ORM;

use App\Entity\Setting;
use App\Entity\Supplier;
use App\Entity\SupplierCoefficient;
use Doctrine\Bundle\FixturesBundle\Fixture;
use Doctrine\Common\Persistence\ObjectManager;
use Payum\Core\Model\GatewayConfig;
use Sylius\Component\Addressing\Model\Province;
use Sylius\Component\Addressing\Model\Zone;
use Sylius\Component\Addressing\Model\ZoneMember;
use Sylius\Component\Core\Model\AdminUserInterface;
use Sylius\Component\Core\Model\Channel;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Model\PaymentMethod;
use Sylius\Component\Core\Model\ShippingMethod;
use Sylius\Component\Locale\Model\Locale;
use Sylius\Component\Product\Model\ProductAssociationType;
use Sylius\Component\Shipping\Model\ShippingCategory;
use Sylius\Component\Taxonomy\Factory\TaxonFactoryInterface;
use Sylius\Component\Taxonomy\Generator\TaxonSlugGenerator;
use Webgriffe\SyliusTableRateShippingPlugin\Entity\ShippingTableRate;

class AppFixtures extends Fixture
{
    /**
     * @var array
     */
    private $services;

    /**
     * @param ObjectManager $manager
     */
    public function load(ObjectManager $manager)
    {
        // Add a locale
        $locale = $this->get('sylius.factory.locale')->createNew();
        $locale->setCode('fr_FR');
        $this->get('sylius.repository.locale')->add($locale);

        // Add a currency
        $currency = $this->get('sylius.factory.currency')->createNew();
        $currency->setCode('EUR');
        $this->get('sylius.repository.currency')->add($currency);

        // Add a country
        $countryRepository = $this->get('sylius.repository.country');
        $countries = ['FR', 'GP', 'GF', 'RE', 'MQ'];

        foreach ($countries as $code) {
            $country = $this->get('sylius.factory.country')->createNew();
            $country->setCode($code);
            $countryRepository->add($country);
        }

        // Add province
        $departments = [
            'FR-01' => 'Ain',
            'FR-02' => 'Aisne',
            'FR-03' => 'Allier',
            'FR-04' => 'Alpes-de-Haute-Provence',
            'FR-05' => 'Hautes-Alpes',
            'FR-06' => 'Alpes-Maritimes',
            'FR-07' => 'Ardèche',
            'FR-08' => 'Ardennes',
            'FR-09' => 'Ariège',
            'FR-10' => 'Aube',
            'FR-11' => 'Aude',
            'FR-12' => 'Aveyron',
            'FR-13' => 'Bouches-du-Rhône',
            'FR-14' => 'Calvados',
            'FR-15' => 'Cantal',
            'FR-16' => 'Charente',
            'FR-17' => 'Charente-Maritime',
            'FR-18' => 'Cher',
            'FR-19' => 'Corrèze',
            'FR-21' => 'Côte-d\'Or',
            'FR-22' => 'Côtes-d\'Armor',
            'FR-23' => 'Creuse',
            'FR-24' => 'Dordogne',
            'FR-25' => 'Doubs',
            'FR-26' => 'Drôme',
            'FR-27' => 'Eure',
            'FR-28' => 'Eure-et-Loir',
            'FR-29' => 'Finistère',
            'FR-2A' => 'Corse-du-Sud',
            'FR-2B' => 'Haute-Corse',
            'FR-30' => 'Gard',
            'FR-31' => 'Haute-Garonne',
            'FR-32' => 'Gers',
            'FR-33' => 'Gironde',
            'FR-34' => 'Hérault',
            'FR-35' => 'Ille-et-Vilaine',
            'FR-36' => 'Indre',
            'FR-37' => 'Indre-et-Loire',
            'FR-38' => 'Isère',
            'FR-39' => 'Jura',
            'FR-40' => 'Landes',
            'FR-41' => 'Loir-et-Cher',
            'FR-42' => 'Loire',
            'FR-43' => 'Haute-Loire',
            'FR-44' => 'Loire-Atlantique',
            'FR-45' => 'Loiret',
            'FR-46' => 'Lot',
            'FR-47' => 'Lot-et-Garonne',
            'FR-48' => 'Lozère',
            'FR-49' => 'Maine-et-Loire',
            'FR-50' => 'Manche',
            'FR-51' => 'Marne',
            'FR-52' => 'Haute-Marne',
            'FR-53' => 'Mayenne',
            'FR-54' => 'Meurthe-et-Moselle',
            'FR-55' => 'Meuse',
            'FR-56' => 'Morbihan',
            'FR-57' => 'Moselle',
            'FR-58' => 'Nièvre',
            'FR-59' => 'Nord',
            'FR-60' => 'Oise',
            'FR-61' => 'Orne',
            'FR-62' => 'Pas-de-Calais',
            'FR-63' => 'Puy-de-Dôme',
            'FR-64' => 'Pyrénées-Atlantiques',
            'FR-65' => 'Hautes-Pyrénées',
            'FR-66' => 'Pyrénées-Orientales',
            'FR-67' => 'Bas-Rhin',
            'FR-68' => 'Haut-Rhin',
            'FR-69' => 'Rhône',
            'FR-70' => 'Haute-Saône',
            'FR-71' => 'Saône-et-Loire',
            'FR-72' => 'Sarthe',
            'FR-73' => 'Savoie',
            'FR-74' => 'Haute-Savoie',
            'FR-75' => 'Paris',
            'FR-76' => 'Seine-Maritime',
            'FR-77' => 'Seine-et-Marne',
            'FR-78' => 'Yvelines',
            'FR-79' => 'Deux-Sèvres',
            'FR-80' => 'Somme',
            'FR-81' => 'Tarn',
            'FR-82' => 'Tarn-et-Garonne',
            'FR-83' => 'Var',
            'FR-84' => 'Vaucluse',
            'FR-85' => 'Vendée',
            'FR-86' => 'Vienne',
            'FR-87' => 'Haute-Vienne',
            'FR-88' => 'Vosges',
            'FR-89' => 'Yonne',
            'FR-90' => 'Territoire de Belfort',
            'FR-91' => 'Essonne',
            'FR-92' => 'Hauts-de-Seine',
            'FR-93' => 'Seine-Saint-Denis',
            'FR-94' => 'Val-de-Marne',
            'FR-95' => 'Val-d\'Oise',
        ];

        foreach ($departments as $code => $department) {
            /** @var Province $province */
            $province = $this->get('sylius.factory.province')->createNew();
            $province->setCode($code);
            $province->setName($department);
            $province->setAbbreviation($department);
            $province->setCountry($countryRepository->findOneBy(['code' => 'FR']));

            $this->get('sylius.repository.province')->add($province);
        }

        // Add a zone
        $zones = [
            'FR' => [
                'name' => 'France',
                'type' => Zone::TYPE_COUNTRY,
                'members' => ['FR'],
            ],
            'FR-DOM' => [
                'name' => 'DOM',
                'type' => Zone::TYPE_COUNTRY,
                'members' => ['GP', 'GF', 'RE', 'MQ'],
            ],
            'FR-DORDOGNE' => [
                'name' => 'Dordogne',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-24'],
            ],
            'FR-CORSE' => [
                'name' => 'Corse',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-2A', 'FR-2B'],
            ],
            'FR-B-ZONE2' => [
                'name' => 'Bernis Zone 2',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-16', 'FR-19', 'FR-33', 'FR-46', 'FR-47', 'FR-87'],
            ],
            'FR-B-ZONE3' => [
                'name' => 'Bernis Zone 3',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-03', 'FR-12', 'FR-15', 'FR-17', 'FR-18', 'FR-23', 'FR-31', 'FR-32', 'FR-36', 'FR-40', 'FR-63', 'FR-79', 'FR-82', 'FR-86'],
            ],
            'FR-B-ZONE4' => [
                'name' => 'Bernis Zone 4',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-09', 'FR-11', 'FR-28', 'FR-34', 'FR-37', 'FR-41', 'FR-44', 'FR-45', 'FR-49', 'FR-58', 'FR-61', 'FR-64', 'FR-65', 'FR-66', 'FR-72', 'FR-81', 'FR-85'],
            ],
            'FR-B-ZONE5' => [
                'name' => 'Bernis Zone 5',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-01', 'FR-02', 'FR-08', 'FR-10', 'FR-14', 'FR-21', 'FR-27', 'FR-30', 'FR-35', 'FR-42', 'FR-43', 'FR-48', 'FR-51', 'FR-53', 'FR-56', 'FR-60', 'FR-69', 'FR-71', 'FR-76', 'FR-80', 'FR-89'],
            ],
            'FR-B-ZONE6' => [
                'name' => 'Bernis Zone 6',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-75', 'FR-77', 'FR-78', 'FR-91', 'FR-92', 'FR-93', 'FR-94', 'FR-95'],
            ],
            'FR-B-ZONE7' => [
                'name' => 'Bernis Zone 7',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-04', 'FR-05', 'FR-06', 'FR-07', 'FR-13', 'FR-22', 'FR-25', 'FR-26', 'FR-29', 'FR-38', 'FR-39', 'FR-50', 'FR-52', 'FR-54', 'FR-55', 'FR-57', 'FR-59', 'FR-62', 'FR-67', 'FR-68', 'FR-70', 'FR-73', 'FR-83', 'FR-84', 'FR-88', 'FR-90'],
            ],
            'FR-BE-ZONE2' => [
                'name' => 'Bernis Express Zone 2',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-16', 'FR-17', 'FR-19', 'FR-33', 'FR-46', 'FR-47', 'FR-87'],
            ],
            'FR-BE-ZONE3' => [
                'name' => 'Bernis Express Zone 3',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-03', 'FR-12', 'FR-15', 'FR-18', 'FR-23', 'FR-31', 'FR-32', 'FR-36', 'FR-40', 'FR-64', 'FR-65', 'FR-79', 'FR-81', 'FR-82', 'FR-86'],
            ],
            'FR-BE-ZONE4' => [
                'name' => 'Bernis Express Zone 4',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-09', 'FR-10', 'FR-11', 'FR-21', 'FR-27', 'FR-28', 'FR-30', 'FR-34', 'FR-37', 'FR-41', 'FR-42', 'FR-43', 'FR-44', 'FR-45', 'FR-48', 'FR-49', 'FR-51', 'FR-53', 'FR-58', 'FR-60', 'FR-61', 'FR-63', 'FR-66', 'FR-69', 'FR-71', 'FR-72',  'FR-85', 'FR-89'],
            ],
            'FR-BE-ZONE5' => [
                'name' => 'Bernis Express Zone 5',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-01', 'FR-07', 'FR-13',  'FR-25', 'FR-26', 'FR-35', 'FR-38', 'FR-39', 'FR-52', 'FR-70', 'FR-84'],
            ],
            'FR-BE-ZONE6' => [
                'name' => 'Bernis Express Zone 6',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-75', 'FR-77', 'FR-78', 'FR-91', 'FR-92', 'FR-93', 'FR-94', 'FR-95'],
            ],
            'FR-BE-ZONE7' => [
                'name' => 'Bernis Express Zone 7',
                'type' => Zone::TYPE_PROVINCE,
                'members' => ['FR-02', 'FR-04', 'FR-05', 'FR-06', 'FR-08', 'FR-14', 'FR-22', 'FR-29', 'FR-50', 'FR-54', 'FR-55', 'FR-56', 'FR-57', 'FR-59', 'FR-62', 'FR-67', 'FR-68', 'FR-73', 'FR-74', 'FR-76', 'FR-80', 'FR-83', 'FR-88', 'FR-90'],
            ],
        ];

        foreach ($zones as $code => $info) {
            $zone = $this->get('sylius.factory.zone')->createNew();
            $zone->setName($info['name']);
            $zone->setCode($code);
            $zone->setType($info['type']);

            foreach ($info['members'] as $member) {
                /** @var ZoneMember $zoneMember */
                $zoneMember = $this->get('sylius.factory.zone_member')->createNew();
                $zoneMember->setCode($member);

                $zone->addMember($zoneMember);
            }

            $this->get('sylius.repository.zone')->add($zone);
        }

        /** @var Channel $channel */
        $channel = $this->get('sylius.factory.channel')->createNew();
        $channel->setName('E-Commerce');
        $channel->setDescription('Site e-commerce');
        $channel->setCode('ECOMMERCE');
        $channel->setEnabled(true);
        $channel->addLocale($locale);
        $channel->setDefaultLocale($locale);
        $channel->addCurrency($currency);
        $channel->setBaseCurrency($currency);
        $channel->setTaxCalculationStrategy('order_items_based');
        $channel->setDefaultTaxZone($this->get('sylius.repository.zone')->findOneBy(['code' => 'FR']));
        $this->get('sylius.repository.channel')->add($channel);

        foreach ([
            'ACCESSORIES' => 'Accessoires',
            'NHP' => 'Nettoyeurs haute pression',
        ] as $code => $name) {
            /** @var ProductAssociationType $productAssociationType */
            $productAssociationType = $this->get('sylius.factory.product_association_type')->createNew();
            $productAssociationType->setName($name);
            $productAssociationType->setCode($code);
            $this->get('sylius.repository.product_association_type')->add($productAssociationType);
        }

        $this->addTax($zones);
        $this->addShippingMethod($channel);
        $this->addPaymentsMethod($channel);
        $this->addTaxons($locale);
        $this->addSuppliers();
        $this->addProductAttributes();
        $this->addUsers($locale);
        $this->addSettings();

        $manager->flush();
    }

    private function addSettings(): void
    {
        $settings = [
            [
                'name' => 'Frais de gestion pour toute commande inférieure à 50€ TTC',
                'code' => Setting::MANAGEMENT_FEES,
                'content' => '5',
                'active' => true,
            ],
            [
                'name' => 'Minimum frais de gestion TTC',
                'code' => Setting::MINIMUM_MANAGEMENT_FEES,
                'content' => '50',
                'active' => true,
            ],
            [
                'name' => 'Minimum frais de port TTC',
                'code' => Setting::MINIMUM_SHIPPING_FEES,
                'content' => '150',
                'active' => true,
            ],
            [
                'name' => 'Minimum commande',
                'code' => Setting::MINIMUM_ORDER,
                'content' => '50',
                'active' => false,
            ],
            [
                'name' => 'Option vacances',
                'code' => Setting::HOLIDAY,
                'content' => 'THS sera fermée du lundi 24 décembre 2018 au mardi 1 janvier 2019 inclus. Réouverture de nos bureaux à 8h30. Les délais de livraison peuvent être rallongés. Merci pour votre compréhension. L\'équipe THS.',
                'active' => false,
            ],
        ];

        foreach ($settings as $info) {
            /** @var Setting $setting */
            $setting = $this->get('app.factory.setting')->createNew();
            $setting->setName($info['name']);
            $setting->setCode($info['code']);
            $setting->setContent($info['content']);
            $setting->setActive($info['active']);

            $this->get('app.repository.setting')->add($setting);
        }
    }

    /**
     * @param ChannelInterface $channel
     */
    private function addShippingMethod(ChannelInterface $channel): void
    {
        $tableRates = [
            [
                'code' => 'FR-LP',
                'name' => 'La Poste France',
                'currency' => 'EUR',
                'fix_price' => 30,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 2,
                        'rate' => 880,
                    ], [
                        'weightLimit' => 5,
                        'rate' => 1270,
                    ], [
                        'weightLimit' => 6,
                        'rate' => 1380,
                    ], [
                        'weightLimit' => 7,
                        'rate' => 1580,
                    ], [
                        'weightLimit' => 8,
                        'rate' => 1710,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 1740,
                    ], [
                        'weightLimit' => 15,
                        'rate' => 2015,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 2495,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 2835,
                    ],
                ],
            ],
            [
                'code' => 'FR-LP-DOM',
                'name' => 'La Poste DOM',
                'currency' => 'EUR',
                'fix_price' => 30,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 0.5,
                        'rate' => 1385,
                    ], [
                        'weightLimit' => 1,
                        'rate' => 1660,
                    ], [
                        'weightLimit' => 2,
                        'rate' => 2271,
                    ], [
                        'weightLimit' => 3,
                        'rate' => 2690,
                    ], [
                        'weightLimit' => 4,
                        'rate' => 2993,
                    ], [
                        'weightLimit' => 5,
                        'rate' => 3680,
                    ], [
                        'weightLimit' => 6,
                        'rate' => 4015,
                    ], [
                        'weightLimit' => 7,
                        'rate' => 4526,
                    ], [
                        'weightLimit' => 8,
                        'rate' => 5237,
                    ], [
                        'weightLimit' => 9,
                        'rate' => 5748,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 6259,
                    ], [
                        'weightLimit' => 15,
                        'rate' => 8459,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 12314,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 16524,
                    ],
                ],
            ],
            [
                'code' => 'FR-CP',
                'name' => 'Chronopost France',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 2,
                        'rate' => 2803,
                    ], [
                        'weightLimit' => 5,
                        'rate' => 3203,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 3929,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 5381,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 6834,
                    ],
                ],
            ],
            [
                'code' => 'FR-B-ZONE1',
                'name' => 'BERNIS ZONE 1',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 15,
                        'rate' => 1529,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 1793,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 1950,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 2196,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 2308,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 2450,
                    ],
                ],
            ],
            [
                'code' => 'FR-B-ZONE2',
                'name' => 'BERNIS ZONE 2',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 10,
                        'rate' => 1815,
                    ], [
                        'weightLimit' => 15,
                        'rate' => 2024,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 2419,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 2500,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 2696,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 2891,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 3237,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 3384,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 3733,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 4078,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 4275,
                    ],
                ],
            ],
            [
                'code' => 'FR-B-ZONE3',
                'name' => 'BERNIS ZONE 3',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 15,
                        'rate' => 2319,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 2839,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 3007,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 3334,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 3684,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 4258,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 4481,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 5010,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 5529,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 5683,
                    ],
                ],
            ],
            [
                'code' => 'FR-B-ZONE4',
                'name' => 'BERNIS ZONE 4',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 9,
                        'rate' => 2145,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 2416,
                    ], [
                        'weightLimit' => 15,
                        'rate' => 2715,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 3195,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 3280,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 3731,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 3856,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 4330,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 4526,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 5019,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 5510,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 5630,
                    ],
                ],
            ],
            [
                'code' => 'FR-B-ZONE5',
                'name' => 'BERNIS ZONE 5',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 9,
                        'rate' => 2201,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 2830,
                    ], [
                        'weightLimit' => 15,
                        'rate' => 3099,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 3103,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 3689,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 4205,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 4331,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 4694,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 4713,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 5502,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 5755,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 6399,
                    ],
                ],
            ],
            [
                'code' => 'FR-B-ZONE6',
                'name' => 'BERNIS ZONE 6',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 10,
                        'rate' => 2334,
                    ], [
                        'weightLimit' => 15,
                        'rate' => 2520,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 3047,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 3597,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 3916,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 4177,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 4553,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 5003,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 5448,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 5645,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 5946,
                    ],
                ],
            ],
            [
                'code' => 'FR-B-ZONE7',
                'name' => 'BERNIS ZONE 7',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 10,
                        'rate' => 2670,
                    ], [
                        'weightLimit' => 15,
                        'rate' => 3105,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 3440,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 3955,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 4590,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 5082,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 5543,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 6175,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 6891,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 7602,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 7832,
                    ],
                ],
            ],
            [
                'code' => 'FR-B-ZONE8',
                'name' => 'BERNIS ZONE 8',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 10,
                        'rate' => 5415,
                    ], [
                        'weightLimit' => 15,
                        'rate' => 6240,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 6673,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 7220,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 8261,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 9118,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 9922,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 10762,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 11978,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 12576,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 13119,
                    ],
                ],
            ],
            [
                'code' => 'FR-BE-ZONE1',
                'name' => 'BERNIS EXPRESS ZONE 1',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 5,
                        'rate' => 1228,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 1362,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 1504,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 1783,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 1884,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 2135,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 2231,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 2343,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 2451,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 2660,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 3115,
                    ],
                ],
            ],
            [
                'code' => 'FR-BE-ZONE2',
                'name' => 'BERNIS EXPRESS ZONE 2',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 5,
                        'rate' => 2589,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 2795,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 3250,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 3702,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 4050,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 4370,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 4545,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 4785,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 5415,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 6079,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 6707,
                    ],
                ],
            ],
            [
                'code' => 'FR-BE-ZONE3',
                'name' => 'BERNIS EXPRESS ZONE 3',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 5,
                        'rate' => 4473,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 5514,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 7088,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 8625,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 10196,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 11737,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 13314,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 14841,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 16406,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 17908,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 19448,
                    ],
                ],
            ],
            [
                'code' => 'FR-BE-ZONE4',
                'name' => 'BERNIS EXPRESS ZONE 4',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 5,
                        'rate' => 4785,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 6288,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 8384,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 10097,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 12085,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 14035,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 15954,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 17908,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 19829,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 21787,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 23772,
                    ],
                ],
            ],
            [
                'code' => 'FR-BE-ZONE5',
                'name' => 'BERNIS EXPRESS ZONE 5',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 5,
                        'rate' => 4920,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 6503,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 8530,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 10223,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 12220,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 14182,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 16074,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 18028,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 19989,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 21904,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 23972,
                    ],
                ],
            ],
            [
                'code' => 'FR-BE-ZONE6',
                'name' => 'BERNIS EXPRESS ZONE 6',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 5,
                        'rate' => 5203,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 7124,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 9567,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 11973,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 14455,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 16863,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 19377,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 21747,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 24193,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 26640,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 29112,
                    ],
                ],
            ],
            [
                'code' => 'FR-BE-ZONE7',
                'name' => 'BERNIS EXPRESS ZONE 7',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 5,
                        'rate' => 5359,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 7341,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 9855,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 12333,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 14889,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 17368,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 19961,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 22428,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 24918,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 27441,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 29983,
                    ],
                ],
            ],
            [
                'code' => 'FR-BE-ZONE8',
                'name' => 'BERNIS EXPRESS ZONE 8',
                'currency' => 'EUR',
                'fix_price' => 50,
                'weightLimitToRate' => [
                    [
                        'weightLimit' => 5,
                        'rate' => 9567,
                    ], [
                        'weightLimit' => 10,
                        'rate' => 12879,
                    ], [
                        'weightLimit' => 20,
                        'rate' => 18258,
                    ], [
                        'weightLimit' => 30,
                        'rate' => 22516,
                    ], [
                        'weightLimit' => 40,
                        'rate' => 26813,
                    ], [
                        'weightLimit' => 50,
                        'rate' => 31066,
                    ], [
                        'weightLimit' => 60,
                        'rate' => 36271,
                    ], [
                        'weightLimit' => 70,
                        'rate' => 40845,
                    ], [
                        'weightLimit' => 80,
                        'rate' => 45242,
                    ], [
                        'weightLimit' => 90,
                        'rate' => 49985,
                    ], [
                        'weightLimit' => 100,
                        'rate' => 54279,
                    ],
                ],
            ],
        ];

        foreach ($tableRates as $info) {
            /** @var ShippingTableRate $tableRate */
            $tableRate = $this->get('webgriffe.factory.shipping_table_rate')->createNew();
            $tableRate->setName($info['name']);
            $tableRate->setCode($info['code']);
            $tableRate->setCurrency($this->get('sylius.repository.currency')->findOneBy(['code' => $info['currency']]));

            foreach ($info['weightLimitToRate'] as $rate) {
                $tableRate->addRate($rate['weightLimit'], $rate['rate'] + $info['fix_price']);
            }

            $this->get('webgriffe.repository.shipping_table_rate')->add($tableRate);
        }

        $shippingCategories = [
            [
                'code' => 'REGULAR',
                'name' => 'Régulier',
                'description' => 'Envoi régulié',
            ],
            [
                'code' => 'OUTSIZE',
                'name' => 'Hors gabarit',
                'description' => 'Envoi hors gabarit',
            ],
        ];

        foreach ($shippingCategories as $category) {
            /** @var ShippingCategory $shippingCategory */
            $shippingCategory = $this->get('sylius.factory.shipping_category')->createNew();
            $shippingCategory->setCode($category['code']);
            $shippingCategory->setName($category['name']);
            $shippingCategory->setDescription($category['description']);

            $this->get('sylius.repository.shipping_category')->add($shippingCategory);
        }

        $shippingMethods = [
            [
                'name' => 'La Poste',
                'code' => 'FR-LP',
                'description' => 'Envoi par la Poste.',
                'zone' => 'FR',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-LP',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ALL,
                'category' => 'REGULAR',
            ], [
                'name' => 'La Poste DOM',
                'code' => 'FR-LP-DOM',
                'description' => 'Envoi par la Poste.',
                'zone' => 'FR-DOM',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-LP-DOM',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ALL,
                'category' => 'REGULAR',
            ], [
                'name' => 'Chronopost',
                'code' => 'FR-CP',
                'description' => 'Envoi par Chronopost.',
                'zone' => 'FR',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-CP',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ALL,
                'category' => 'REGULAR',
            ], [
                'name' => 'Bernis',
                'code' => 'FR-B-DORDOGNE',
                'description' => 'Envoi par Bernis.',
                'zone' => 'FR-DORDOGNE',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-B-ZONE1',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],  [
                'name' => 'Bernis',
                'code' => 'FR-B-ZONE2',
                'description' => 'Envoi par Bernis.',
                'zone' => 'FR-B-ZONE2',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-B-ZONE2',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],   [
                'name' => 'Bernis',
                'code' => 'FR-B-ZONE3',
                'description' => 'Envoi par Bernis.',
                'zone' => 'FR-B-ZONE3',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-B-ZONE3',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],   [
                'name' => 'Bernis',
                'code' => 'FR-B-ZONE4',
                'description' => 'Envoi par Bernis.',
                'zone' => 'FR-B-ZONE4',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-B-ZONE4',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],   [
                'name' => 'Bernis',
                'code' => 'FR-B-ZONE5',
                'description' => 'Envoi par Bernis.',
                'zone' => 'FR-B-ZONE5',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-B-ZONE5',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],   [
                'name' => 'Bernis',
                'code' => 'FR-B-ZONE6',
                'description' => 'Envoi par Bernis.',
                'zone' => 'FR-B-ZONE6',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-B-ZONE6',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],   [
                'name' => 'Bernis',
                'code' => 'FR-B-ZONE7',
                'description' => 'Envoi par Bernis.',
                'zone' => 'FR-B-ZONE7',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-B-ZONE7',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ], [
                'name' => 'Bernis',
                'code' => 'FR-B-CORSE',
                'description' => 'Envoi par Bernis.',
                'zone' => 'FR-CORSE',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-B-ZONE8',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ], [
                'name' => 'Bernis Express',
                'code' => 'FR-BE-DORDOGNE',
                'description' => 'Envoi par Bernis Express.',
                'zone' => 'FR-DORDOGNE',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-BE-ZONE1',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],  [
                'name' => 'Bernis Express',
                'code' => 'FR-BE-ZONE2',
                'description' => 'Envoi par Bernis Express.',
                'zone' => 'FR-BE-ZONE2',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-BE-ZONE2',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],   [
                'name' => 'Bernis Express',
                'code' => 'FR-BE-ZONE3',
                'description' => 'Envoi par Bernis Express.',
                'zone' => 'FR-BE-ZONE3',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-BE-ZONE3',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],   [
                'name' => 'Bernis Express',
                'code' => 'FR-BE-ZONE4',
                'description' => 'Envoi par Bernis Express.',
                'zone' => 'FR-BE-ZONE4',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-BE-ZONE4',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],   [
                'name' => 'Bernis Express',
                'code' => 'FR-BE-ZONE5',
                'description' => 'Envoi par Bernis Express.',
                'zone' => 'FR-BE-ZONE5',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-BE-ZONE5',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],   [
                'name' => 'Bernis Express',
                'code' => 'FR-BE-ZONE6',
                'description' => 'Envoi par Bernis Express.',
                'zone' => 'FR-BE-ZONE6',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-BE-ZONE6',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],   [
                'name' => 'Bernis Express',
                'code' => 'FR-BE-ZONE7',
                'description' => 'Envoi par Bernis Express.',
                'zone' => 'FR-BE-ZONE7',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-BE-ZONE7',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ], [
                'name' => 'Bernis Express',
                'code' => 'FR-BE-CORSE',
                'description' => 'Envoi par Bernis Express.',
                'zone' => 'FR-CORSE',
                'calculator' => 'table_rate',
                'tale_rate' => 'FR-BE-ZONE8',
                'category_requirement' => ShippingMethod::CATEGORY_REQUIREMENT_MATCH_ANY,
                'category' => 'OUTSIZE',
            ],
        ];

        foreach ($shippingMethods as $shipping) {
            /** @var ShippingMethod $shippingMethod */
            $shippingMethod = $this->get('sylius.factory.shipping_method')->createNew();
            $shippingMethod->addChannel($channel);
            $shippingMethod->setCode($shipping['code']);
            $shippingMethod->setDescription($shipping['description']);
            $shippingMethod->setName($shipping['name']);
            $shippingMethod->setEnabled(true);
            $shippingMethod->setCalculator($shipping['calculator']);
            $shippingMethod->setCategoryRequirement($shipping['category_requirement']);
            $shippingMethod->setCategory($this->get('sylius.repository.shipping_category')->findOneBy(['code' => $shipping['category']]));
            $shippingMethod->setZone($this->get('sylius.repository.zone')->findOneBy(['code' => $shipping['zone']]));
            $shippingMethod->setConfiguration([
                'ECOMMERCE' => [
                    'table_rate' => $this->get('webgriffe.repository.shipping_table_rate')->findOneBy(['code' => $shipping['tale_rate']]),
                ],
            ]);
            $shippingMethod->setTaxCategory($this->get('sylius.repository.tax_category')->findOneBy(['code' => 'SALE20']));

            $this->get('sylius.repository.shipping_method')->add($shippingMethod);
        }
    }

    /**
     * @param array $zones
     */
    private function addTax(array $zones): void
    {
        // Add a tax category
        $taxCategories = [
            'SALE20' => [
                'name' => 'Vente 20%',
                'description' => 'Vente 20%',
            ],
            'SALE196' => [
                'name' => 'Vente 19,6%',
                'description' => 'Vente 19,6%',
            ],
        ];

        foreach ($taxCategories as $code => $tax) {
            $taxCategory = $this->get('sylius.factory.tax_category')->createNew();
            $taxCategory->setName($tax['name']);
            $taxCategory->setCode($code);
            $taxCategory->setDescription($tax['description']);
            $this->get('sylius.repository.tax_category')->add($taxCategory);
        }

        // Add tax rate
        $taxes = [
            [
                'code' => 'NORMAL20',
                'name' => 'Taux normal 20%',
                'amount' => 0.2,
                'tax_category' => 'SALE20',
            ],
            [
                'code' => 'NORMAL196',
                'name' => 'Taux normal 19,6%',
                'amount' => 0.196,
                'tax_category' => 'SALE196',
            ],
        ];

        foreach ($taxes as $tax) {
            foreach ($zones as $code => $zone) {
                $taxRate = $this->get('sylius.factory.tax_rate')->createNew();
                $taxRate->setCode($tax['code'].'-'.$code);
                $taxRate->setName($tax['name']);
                $taxRate->setAmount($tax['amount']);
                $taxRate->setIncludedInPrice(false);
                $taxRate->setZone($this->get('sylius.repository.zone')->findOneBy(['code' => $code]));
                $taxRate->setCalculator('default');
                $taxRate->setCategory($this->get('sylius.repository.tax_category')->findOneBy(['code' => $tax['tax_category']]));

                $this->get('sylius.repository.tax_rate')->add($taxRate);
            }
        }
    }

    /**
     * @param $channel
     */
    private function addPaymentsMethod($channel): void
    {
        /** @var GatewayConfig $offlineGateway */
        $offlineGateway = $this->get('sylius.factory.gateway_config')->createNew();
        $offlineGateway->setGatewayName('offline');
        $offlineGateway->setFactoryName('offline');

        /** @var GatewayConfig $citelisGateway */
        $citelisGateway = $this->get('sylius.factory.gateway_config')->createNew();
        $citelisGateway->setGatewayName('citelis');
        $citelisGateway->setFactoryName('citelis');

        /*
         * For dev only.
         */
        $citelisGateway->setConfig([
            'merchant_id' => '029800266211111',
            'currency_code' => 978,
            'cgi_bin_dir' => '/home/yenkong/Sites/ths/cgi-bin',
            'merchant_country' => 'fr',
        ]);

        $paymentsMethod = [
            'CHQ' => [
                'name' => 'Chèque',
                'description' => 'Paiement par chèque',
                'instructions' => 'Nous vous prions de bien vouloir nous retourner le détail de la commande ainsi qu\'un chèque du montant de la commande à l\'ordre de "Technique Haute pression & Sablage". Nous procéderons au traitement de celle-ci dès la réception du chèque.',
                'gateway' => $offlineGateway,
            ],
            'MANDAT' => [
                'name' => 'Mandat administratif',
                'description' => 'Paiement par mandat administratif',
                'instructions' => 'Vous avez choisi le paiement par mandat administratif. Pour cela, merci d’imprimer cette commande et de la joindre à votre bon de commande et de nous envoyer ces 2 documents (Commande du site haute-pression + Votre bon de commande) par mail : magasin@thsgroupe.com ou par fax : (+33 0)5 53 58 39 98. Nous procéderons au traitement de celle-ci dès la réception des documents.',
                'gateway' => $offlineGateway,
            ],
            'CITELIS' => [
                'name' => 'Carte bleue',
                'description' => 'Paiement par carte bleue',
                'instructions' => '',
                'gateway' => $citelisGateway,
            ],
        ];

        foreach ($paymentsMethod as $code => $_paymentMethod) {
            /** @var PaymentMethod $paymentMethod */
            $paymentMethod = $this->get('sylius.factory.payment_method')->createNew();
            $paymentMethod->setGatewayConfig($_paymentMethod['gateway']);
            $paymentMethod->setCode($code);
            $paymentMethod->setEnabled(true);
            $paymentMethod->setName($_paymentMethod['name']);
            $paymentMethod->setDescription($_paymentMethod['description']);
            $paymentMethod->setInstructions($_paymentMethod['instructions']);
            $paymentMethod->addChannel($channel);

            $this->get('sylius.repository.payment_method')->add($paymentMethod);
        }
    }

    /**
     * @param Locale $locale
     */
    private function addTaxons(Locale $locale): void
    {
        /** @var TaxonSlugGenerator $taxonSlugGenerator */
        $taxonSlugGenerator = $this->get('sylius.generator.taxon_slug');
        /** @var TaxonFactoryInterface $taxonFactory */
        $taxonFactory = $this->get('sylius.factory.taxon');
        $taxonRepository = $this->get('sylius.repository.taxon');

        $taxonArticle = $taxonFactory->createNew();
        $taxonArticle->setName('Article');
        $taxonArticle->setPosition(0);
        $taxonArticle->setCode('ARTICLE');
        $taxonArticle->setCurrentLocale($locale->getCode());
        $taxonArticle->setFallbackLocale($locale->getCode());
        $taxonArticle->setSlug($taxonSlugGenerator->generate($taxonArticle));
        $taxonRepository->add($taxonArticle);

        $taxonAccessoires = $taxonFactory->createNew();
        $taxonAccessoires->setName('Accessoires');
        $taxonAccessoires->setPosition(0);
        $taxonAccessoires->setCode('ACCESSOIRES');
        $taxonAccessoires->setCurrentLocale($locale->getCode());
        $taxonAccessoires->setFallbackLocale($locale->getCode());
        $taxonAccessoires->setSlug($taxonSlugGenerator->generate($taxonAccessoires));
        $taxonRepository->add($taxonAccessoires);

        $taxonPiecesDetachees = $taxonFactory->createNew();
        $taxonPiecesDetachees->setName('Pièces détachées');
        $taxonPiecesDetachees->setPosition(0);
        $taxonPiecesDetachees->setCode('PIECESDETACHEES');
        $taxonPiecesDetachees->setCurrentLocale($locale->getCode());
        $taxonPiecesDetachees->setFallbackLocale($locale->getCode());
        $taxonPiecesDetachees->setSlug($taxonSlugGenerator->generate($taxonPiecesDetachees));
        $taxonRepository->add($taxonPiecesDetachees);

        $listArticles = [
            'Catalogues et articles spéciaux',
        ];

        foreach ($listArticles as $key => $element) {
            $taxon = $taxonFactory->createNew();
            $taxon->setName($element);
            $taxon->setCode(trim(strtoupper($element)));
            $taxon->setParent($taxonArticle);
            $taxon->setPosition($key);
            $taxon->setCurrentLocale($locale->getCode());
            $taxon->setFallbackLocale($locale->getCode());
            $taxon->setSlug($taxonSlugGenerator->generate($taxon, $locale->getCode()));
            $taxonRepository->add($taxon);
        }

        $listAccessoires = [
            'Flexibles debouchage canalisations',
            'Kit de sablage',
            'Brosses de lavage',
            'Buses debouchage canalisations',
            'Buses haute pression',
            'Huile de pompe HP',
            'Lances haute pression',
            'Manometres',
            'Pistolets haute pression',
            'Pompes haute pression',
            'Tuyaux haute pression',
            'Rotabuses',
            'Condensateur',
            'Interrupteur',
            'Equipement électrique',
            'Equipement brûleurs et serpentins',
            'Canons à mousse',
            'Kit de réparation pompe haute pression',
            'Produits chimique Kranzle',
            'Accessoires Kranzle',
            'Accessoires RM',
            'Accessoires Karcher',
            'Transport',
            'Accessoires Numatic',
            'Pompes haute pression Comet',
            'Pompes Comet Agriculture',
        ];

        foreach ($listAccessoires as $key => $element) {
            $taxon = $taxonFactory->createNew();
            $taxon->setName($element);
            $taxon->setCode(trim(strtoupper($element)));
            $taxon->setParent($taxonAccessoires);
            $taxon->setPosition($key);
            $taxon->setCurrentLocale($locale->getCode());
            $taxon->setFallbackLocale($locale->getCode());
            $taxon->setSlug($taxonSlugGenerator->generate($taxon, $locale->getCode()));
            $taxonRepository->add($taxon);
        }

        $listPiecesDetachees = [
            'Raccords et coupleurs haute pression',
            'Pieces detachees Kranzle',
            'Pieces detachees Comet',
            'Pieces detachees RM',
            'Pieces detachees ICA',
            'Pieces detachees Numatic',
            'Pieces detachees Karcher',
            'Pieces detachees Pratissoli',
            'Pieces detachees Hawk',
            'Pieces detachees Comet Agriculture',
            'Pieces detachees PA',
            'Pieces detachees UDOR',
        ];

        foreach ($listPiecesDetachees as $key => $element) {
            $taxon = $taxonFactory->createNew();
            $taxon->setName($element);
            $taxon->setCode(trim(strtoupper($element)));
            $taxon->setParent($taxonPiecesDetachees);
            $taxon->setPosition($key);
            $taxon->setCurrentLocale($locale->getCode());
            $taxon->setFallbackLocale($locale->getCode());
            $taxon->setSlug($taxonSlugGenerator->generate($taxon, $locale->getCode()));
            $taxonRepository->add($taxon);
        }

        /**
         * Add taxons for NHP and Pompe.
         */
        $taxonNhp = $taxonFactory->createNew();
        $taxonNhp->setName('Nettoyeur Haute Pression');
        $taxonNhp->setPosition(0);
        $taxonNhp->setCode('NHP');
        $taxonNhp->setCurrentLocale($locale->getCode());
        $taxonNhp->setFallbackLocale($locale->getCode());
        $taxonNhp->setSlug($taxonSlugGenerator->generate($taxonNhp));
        $taxonRepository->add($taxonNhp);

        $list = [
            'Alto/Nilfisk',
            'Comet',
            'Dimaco',
            'Dynajet',
            'Heurtaux',
            'ICA',
            'Interpump',
            'Ipc / Portotecnica',
            'Karcher',
            'Kranzle',
            'Lindhaus',
            'Oertzen',
            'THS',
            'Van Damme',
        ];

        foreach ($list as $key => $element) {
            $taxon = $taxonFactory->createNew();
            $taxon->setName($element);
            $taxon->setCode('N_'.str_replace([' ', '/'], '', trim(strtoupper($element))));
            $taxon->setParent($taxonNhp);
            $taxon->setPosition($key);
            $taxon->setCurrentLocale($locale->getCode());
            $taxon->setFallbackLocale($locale->getCode());
            $taxon->setSlug($taxonSlugGenerator->generate($taxon, $locale->getCode()));
            $taxonRepository->add($taxon);
        }

        $taxonPompe = $taxonFactory->createNew();
        $taxonPompe->setName('Pompe');
        $taxonPompe->setPosition(0);
        $taxonPompe->setCode('POMPE');
        $taxonPompe->setCurrentLocale($locale->getCode());
        $taxonPompe->setFallbackLocale($locale->getCode());
        $taxonPompe->setSlug($taxonSlugGenerator->generate($taxonPompe));
        $taxonRepository->add($taxonPompe);

        $list = [
            'Annovi Reverberi',
            'Bertolini',
            'Cat',
            'Comet',
            'Comet agriculture',
            'Fiorentini',
            'Hawk',
            'Interpump',
            'Karcher',
            'Kranzle',
            'Pratissoli',
            'Rumaucourt',
            'Speck',
            'Tractoricambi',
            'Udor',
            'Van Damme',
        ];

        foreach ($list as $key => $element) {
            $taxon = $taxonFactory->createNew();
            $taxon->setName($element);
            $taxon->setCode('P_'.str_replace(' ', '', trim(strtoupper($element))));
            $taxon->setParent($taxonPompe);
            $taxon->setPosition($key);
            $taxon->setCurrentLocale($locale->getCode());
            $taxon->setFallbackLocale($locale->getCode());
            $taxon->setSlug($taxonSlugGenerator->generate($taxon, $locale->getCode()));
            $taxonRepository->add($taxon);
        }
    }

    private function addSuppliers(): void
    {
        $supplierFactory = $this->get('app.factory.supplier');
        $supplierCoefficientFactory = $this->get('app.factory.supplier_coefficient');
        $supplierRepository = $this->get('app.repository.supplier');

        $list = [
            [
                'code' => 'RM',
                'name' => 'Fournisseur RM - Pièces et accessoires',
                'description' => 'Fournisseur RM',
                'coefficients' => [
                    [
                        'coefficient' => 3.0,
                        'minimum_price' => 0.01,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 2.1,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 1.7,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 5.01,
                        'maximum_price' => 9.0,
                    ],
                    [
                        'coefficient' => 1.4,
                        'minimum_price' => 9.01,
                        'maximum_price' => 15.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 15.01,
                        'maximum_price' => 25.0,
                    ],
                ],
            ],
            [
                'code' => 'KRANZLE',
                'name' => 'Fournisseur KRANZLE - Pièces et accessoires',
                'description' => 'Fournisseur KRANZLE',
                'coefficients' => [
                    [
                        'coefficient' => 4.1,
                        'minimum_price' => 0.01,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 3.5,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.8,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 10.01,
                        'maximum_price' => 26.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 26.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'NUMATIC',
                'name' => 'Fournisseur NUMATIC - Pièces et accessoires',
                'description' => 'Fournisseur NUMATIC',
                'coefficients' => [
                    [
                        'coefficient' => 5.0,
                        'minimum_price' => 0.01,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 3.0,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 2.5,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.8,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 10.01,
                        'maximum_price' => 26.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 26.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'ICA',
                'name' => 'Fournisseur ICA - Pièces et accessoires',
                'description' => 'Fournisseur ICA',
                'coefficients' => [
                    [
                        'coefficient' => 4.1,
                        'minimum_price' => 0.01,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 3.5,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.8,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 10.01,
                        'maximum_price' => 25.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 25.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'COMET',
                'name' => 'COMET INDUSTRIE - Pièces et accessoires',
                'description' => 'COMET INDUSTRIE - Pièces et accessoires',
                'coefficients' => [
                    [
                        'coefficient' => 4.1,
                        'minimum_price' => 0.01,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 3.5,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.8,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 10.01,
                        'maximum_price' => 25.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 25.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'PA',
                'name' => 'Fournisseur PA - Pièces et accessoires',
                'description' => 'Fournisseur PA',
                'coefficients' => [
                    [
                        'coefficient' => 3.0,
                        'minimum_price' => 0.01,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 2.1,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 1.7,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 5.01,
                        'maximum_price' => 9.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 9.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'LINDHAUS',
                'name' => 'Fournisseur LINDHAUS - Pièces et accessoires',
                'description' => 'Fournisseur LINDHAUS',
                'coefficients' => [
                    [
                        'coefficient' => 5.0,
                        'minimum_price' => 0.01,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 4.0,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 3.0,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 2.5,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 10.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'KARCHER',
                'name' => 'Fournisseur KARCHER',
                'description' => 'Fournisseur KARCHER',
                'coefficients' => [
                    [
                        'coefficient' => 4.1,
                        'minimum_price' => 0.01,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 3.5,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.8,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 10.01,
                        'maximum_price' => 26.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 26.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'Tractoricambi',
                'name' => 'Fournisseur Tractoricambi - Pièces et accessoires',
                'description' => 'Fournisseur Tractoricambi',
                'coefficients' => [
                    [
                        'coefficient' => 3.1,
                        'minimum_price' => 0.01,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.3,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 10.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'UDOR',
                'name' => 'Fournisseur UDOR - Pièces et accessoires',
                'description' => 'Fournisseur UDOR',
                'coefficients' => [
                    [
                        'coefficient' => 3.1,
                        'minimum_price' => 0.01,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.3,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 10.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'BERTOLINI',
                'name' => 'Fournisseur BERTOLINI - Pièces et accessoires',
                'description' => 'Fournisseur BERTOLINI',
                'coefficients' => [
                    [
                        'coefficient' => 3.1,
                        'minimum_price' => 0.1,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.3,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 10.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'PRATISSOLI',
                'name' => 'Fournisseur PRATISSOLI - Pièces et accessoires',
                'description' => 'Fournisseur PRATISSOLI',
                'coefficients' => [
                    [
                        'coefficient' => 3.1,
                        'minimum_price' => 0.1,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.3,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 10.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'SIRMAC',
                'name' => 'Fournisseur SIRMAC - Pièces et accessoires',
                'description' => 'Fournisseur SIRMAC',
                'coefficients' => [
                    [
                        'coefficient' => 4.1,
                        'minimum_price' => 0.1,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 3.5,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.8,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 10.01,
                        'maximum_price' => 26.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 26.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'FIORENTINI',
                'name' => 'Fournisseur FIORENTINI - Pièces et accessoires',
                'description' => 'Fournisseur FIORENTINI',
                'coefficients' => [
                    [
                        'coefficient' => 5.0,
                        'minimum_price' => 0.1,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 4.0,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 3.0,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 2.5,
                        'minimum_price' => 5.01,
                        'maximum_price' => 25.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 25.01,
                        'maximum_price' => 60.0,
                    ],
                ],
            ],
            [
                'code' => 'COMET AGRICULTURE',
                'name' => 'COMET AGRICULTURE - Pièces et accessoires',
                'description' => 'AGRICULTURE - Pièces et accessoires',
                'coefficients' => [
                    [
                        'coefficient' => 4.1,
                        'minimum_price' => 0.1,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 3.5,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.8,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 10.01,
                        'maximum_price' => 25.0,
                    ],
                    [
                        'coefficient' => 1.2,
                        'minimum_price' => 25.01,
                        'maximum_price' => 50.0,
                    ],
                ],
            ],
            [
                'code' => 'DIMACO',
                'name' => 'Fournisseur DIMACO - Pièces et accessoires',
                'description' => 'Fournisseur DIMACO',
                'coefficients' => [
                    [
                        'coefficient' => 4.1,
                        'minimum_price' => 0.1,
                        'maximum_price' => 1.0,
                    ],
                    [
                        'coefficient' => 3.5,
                        'minimum_price' => 1.01,
                        'maximum_price' => 2.0,
                    ],
                    [
                        'coefficient' => 2.0,
                        'minimum_price' => 2.01,
                        'maximum_price' => 5.0,
                    ],
                    [
                        'coefficient' => 1.8,
                        'minimum_price' => 5.01,
                        'maximum_price' => 10.0,
                    ],
                    [
                        'coefficient' => 1.5,
                        'minimum_price' => 10.01,
                        'maximum_price' => 26.0,
                    ],
                ],
            ],
        ];

        foreach ($list as $element) {
            /** @var Supplier $supplier */
            $supplier = $supplierFactory->createNew();
            $supplier->setName($element['name']);
            $supplier->setCode($element['code']);
            $supplier->setDescription($element['description']);

            foreach ($element['coefficients'] as $coefficient) {
                /** @var SupplierCoefficient $supplierCoefficient */
                $supplierCoefficient = $supplierCoefficientFactory->createNew();
                $supplierCoefficient->setCoefficient($coefficient['coefficient']);
                $supplierCoefficient->setMinimumPrice($coefficient['minimum_price']);
                $supplierCoefficient->setMaximumPrice($coefficient['maximum_price']);
                $supplier->addSupplierCoefficient($supplierCoefficient);
            }

            $supplierRepository->add($supplier);
        }
    }

    private function addProductAttributes()
    {
        /** @var AttributeFactoryInterface $attributeFactory */
        $attributeFactory = $this->get('sylius.factory.product_attribute');
        $repository = $this->get('sylius.repository.product_attribute');

        $attributes = [
            ['name' => 'Pression', 'code' => 'pressure', 'typed' => 'text'],
            ['name' => 'Débit litre minute', 'code' => 'flow_liters_minute', 'typed' => 'text'],
            ['name' => 'Débit litre heure', 'code' => 'flow_liters_hour', 'typed' => 'text'],
            ['name' => 'Vitesse de rotation', 'code' => 'rotation_speed', 'typed' => 'text'],
            ['name' => 'Eau froide', 'code' => 'cold_water', 'typed' => 'checkbox'],
            ['name' => 'Eau chaude', 'code' => 'hot_water', 'typed' => 'checkbox'],
            ['name' => 'Vapeur', 'code' => 'steam', 'typed' => 'checkbox'],
            ['name' => 'Température', 'code' => 'temperature', 'typed' => 'text'],
            ['name' => 'Poste fixe', 'code' => 'fixed_station', 'typed' => 'checkbox'],
            ['name' => 'Groupe moteur pompe', 'code' => 'pump_motor_group', 'typed' => 'checkbox'],
            ['name' => 'Moteur thermique', 'code' => 'thermal_motor', 'typed' => 'checkbox'],
            ['name' => 'Moteur thermique information', 'code' => 'thermal_motor_information', 'typed' => 'text'],
            ['name' => 'Poids', 'code' => 'weight', 'typed' => 'percent'],
            ['name' => 'Spécificité', 'code' => 'specificity', 'typed' => 'text'],
            ['name' => 'Dimension a', 'code' => 'dimension_a', 'typed' => 'text'],
            ['name' => 'Dimension b', 'code' => 'dimension_b', 'typed' => 'text'],
            ['name' => 'Matière', 'code' => 'material', 'typed' => 'text'],
            ['name' => 'Pression maximum', 'code' => 'maximum_pressure', 'typed' => 'text'],
            ['name' => 'Température maximum', 'code' => 'maximum_temperature', 'typed' => 'text'],
            ['name' => 'Diamètre nominal', 'code' => 'nominal_diameter', 'typed' => 'text'],
            ['name' => 'Référence métrique entrée', 'code' => 'metric_reference_input', 'typed' => 'text'],
            ['name' => 'Sexe entrée', 'code' => 'sex_input', 'typed' => 'text'],
            ['name' => 'Sexe sortie', 'code' => 'sex_output', 'typed' => 'text'],
            ['name' => 'Précision supplémentaire entrée', 'code' => 'additional_precision_input', 'typed' => 'text'],
            ['name' => 'Référence métrique sortie', 'code' => 'metric_reference_output', 'typed' => 'text'],
            ['name' => 'Précision supplémentaire sortie', 'code' => 'additional_precision_output', 'typed' => 'text'],
            ['name' => 'Calibre pièce', 'code' => 'caliber_piece', 'typed' => 'text'],
            ['name' => 'Débit maximum', 'code' => 'maximum_flow', 'typed' => 'text'],
        ];

        foreach ($attributes as $data) {
            /** @var AttributeInterface $attribute */
            $attribute = $attributeFactory->createTyped($data['typed']);

            $attribute->setName($data['name']);
            $attribute->setCode($data['code']);

            $repository->add($attribute);
        }
    }

    /**
     * @param Locale $locale
     */
    private function addUsers(Locale $locale): void
    {
        /** @var AdminUserInterface $admin */
        $admin = $this->get('sylius.factory.admin_user')->createNew();
        $admin->setEmail('yenkong@lybliamay.fr');
        $admin->setPlainPassword('sylius');
        $admin->setLocaleCode($locale->getCode());
        $admin->setEnabled(true);
        $this->get('sylius.repository.admin_user')->add($admin);
    }

    /**
     * @param string $id
     *
     * @return mixed|object
     */
    private function get(string $id)
    {
        return isset($this->services[$id])
            ? $this->services[$id]
            : $this->services[$id] = $this->container->get($id);
    }
}
