<?php

declare(strict_types=1);

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Sylius\Component\Core\Model\AdjustmentInterface;
use Sylius\Component\Core\Model\AdminUserInterface;
use Sylius\Component\Core\Model\Order as BaseOrder;
use Sylius\Component\Core\Model\OrderItem;
use Sylius\Component\Core\Model\PaymentMethod;

/**
 * @ORM\Table(name="sylius_order")
 * @ORM\Entity
 */
class Order extends BaseOrder
{
    const TYPE_DEVIS = 1;
    const TYPE_PANIER = 2;
    const TYPE_TICKET = 3;

    const STATE_OFFERED = 'offered';

    /**
     * @ORM\Column(name="token_order", type="string", length=255, nullable=true)
     *
     * @var string|null
     */
    protected $tokenOrder;

    /**
     * @ORM\Column(type="smallint", nullable=true)
     *
     * @var int
     */
    protected $type;

    /**
     * @ORM\Column(type="decimal", precision=10, scale=2 , nullable=true)
     *
     * @var float
     */
    protected $weight;

    /**
     * @ORM\Column(name="order_token", type="string", length=255, nullable=true)
     *
     * @var string|null
     */
    protected $orderToken;

    /**
     * @ORM\Column(name="quotation_token", type="string", length=255,
     *                                     nullable=true)
     *
     * @var string|null
     */
    protected $quotationToken;

    /**
     * @ORM\Column(name="quotation_pdf", type="string", length=255,
     *                                   nullable=true)
     *
     * @var string|null
     */
    protected $quotationPdf;

    /**
     * @ORM\Column(name="id_user_tracking", type="integer", nullable=true)
     *
     * @var int
     */
    protected $idUserTracking;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\AdminUser")
     * @ORM\JoinColumn(name="id_salesman", referencedColumnName="id")
     *
     * @var AdminUserInterface|null
     */
    private $salesman;

    /**
     * @ORM\Column(name="mail_order", type="boolean", nullable=true)
     *
     * @var bool
     */
    protected $mailOrder;

    /**
     * @ORM\Column(name="mail_colissimo", type="boolean", nullable=true)
     *
     * @var bool
     */
    protected $mailColissimo;

    /**
     * @ORM\Column(name="old_order_id", type="integer", nullable=true)
     *
     * @var int
     */
    protected $oldOrderId;

    /**
     * @ORM\Column(name="email_link_clicked", type="boolean", nullable=false)
     *
     * @var bool
     */
    protected $emailLinkClicked = false;

    public function __construct()
    {
        $this->type = self::TYPE_PANIER;

        parent::__construct();
    }

    public function getIdUserTracking(): int
    {
        return $this->idUserTracking;
    }

    /**
     * @return Order
     */
    public function setIdUserTracking(int $idUserTracking): self
    {
        $this->idUserTracking = $idUserTracking;

        return $this;
    }

    public function getOrderToken(): ?string
    {
        return $this->orderToken;
    }

    /**
     * @return Order
     */
    public function setOrderToken(?string $orderToken): self
    {
        $this->orderToken = $orderToken;

        return $this;
    }

    public function isMailOrder(): bool
    {
        return $this->mailOrder;
    }

    /**
     * @return Order
     */
    public function setMailOrder(bool $mailOrder): self
    {
        $this->mailOrder = $mailOrder;

        return $this;
    }

    public function getOldOrderId(): int
    {
        return $this->oldOrderId;
    }

    /**
     * @return Order
     */
    public function setOldOrderId(int $oldOrderId): self
    {
        $this->oldOrderId = $oldOrderId;

        return $this;
    }

    public function getTokenOrder(): ?string
    {
        return $this->tokenOrder;
    }

    /**
     * @return Order
     */
    public function setTokenOrder(?string $tokenOrder): self
    {
        $this->tokenOrder = $tokenOrder;

        return $this;
    }

    public function getQuotationToken(): ?string
    {
        return $this->quotationToken;
    }

    /**
     * @return Order
     */
    public function setQuotationToken(?string $quotationToken): self
    {
        $this->quotationToken = $quotationToken;

        return $this;
    }

    public function getQuotationPdf(): ?string
    {
        return $this->quotationPdf;
    }

    /**
     * @return Order
     */
    public function setQuotationPdf(?string $quotationPdf): self
    {
        $this->quotationPdf = $quotationPdf;

        return $this;
    }

    public function getType(): int
    {
        return $this->type;
    }

    /**
     * @return Order
     */
    public function setType(int $type): self
    {
        $this->type = $type;

        return $this;
    }

    public function getWeight(): float
    {
        return $this->weight;
    }

    /**
     * @return Order
     */
    public function setWeight(float $weight): self
    {
        $this->weight = $weight;

        return $this;
    }

    public function isMailColissimo(): bool
    {
        return $this->mailColissimo;
    }

    /**
     * @return Order
     */
    public function setMailColissimo(bool $mailColissimo): self
    {
        $this->mailColissimo = $mailColissimo;

        return $this;
    }

    /**
     * @return Order
     */
    public function setItemsTotal(int $itemsTotal): self
    {
        $this->itemsTotal = $itemsTotal;

        return $this;
    }

    /**
     * @return Order
     */
    public function setTotal(int $total): self
    {
        $this->total = $total;

        return $this;
    }

    public function getLastPaymentMethodCode(): ?string
    {
        if (0 === $this->getPayments()->count()) {
            return null;
        }

        $paidPayment = $this->getPaidPaymentMethodCode();

        if ($paidPayment) {
            return $paidPayment;
        }

        /** @var Payment $payment */
        $payment = $this->getPayments()->last();

        return $payment->getMethod()->getCode();
    }

    public function getPaidPayment(): ?Payment
    {
        $payments = $this->getPayments()->filter(function (Payment $payment) {
            return $payment->getState() && Payment::STATE_COMPLETED === $payment->getState();
        });

        if (0 === $payments->count()) {
            return null;
        }

        return $payments->first();
    }

    public function getPaidPaymentMethod(): ?PaymentMethod
    {
        $payment = $this->getPaidPayment();

        return is_null($payment) ? null : $payment->getMethod();
    }

    public function getPaidPaymentMethodCode(): ?string
    {
        $payment = $this->getPaidPayment();

        return is_null($payment) ? null : $payment->getMethod()->getCode();
    }

    public function removePayments(): void
    {
        $this->payments->clear();
    }

    public function getSalesman(): ?AdminUserInterface
    {
        return $this->salesman;
    }

    /**
     * @return Order
     */
    public function setSalesman(?AdminUserInterface $salesman): self
    {
        $this->salesman = $salesman;

        return $this;
    }

    public function getManagementFees(): int
    {
        return $this->getAdjustmentAmount(Adjustment::MANAGEMENT_FEES_ADJUSTMENT);
    }

    public function getAdjustmentAmount(string $type): int
    {
        $taxTotal = 0;

        foreach ($this->getAdjustments($type) as $taxAdjustment) {
            $taxTotal += $taxAdjustment->getAmount();
        }

        return $taxTotal;
    }

    public function getShippingTotalWithoutTax(): int
    {
        $shippingTotal = $this->getShippingTotal();
        $shippingTotal -= $this->getAdjustmentsTotal(AdjustmentInterface::TAX_ADJUSTMENT);

        return $shippingTotal;
    }

    public function getTotalWithoutTax(): int
    {
        $total = 0;

        /** @var OrderItem $item */
        foreach ($this->getItems() as $item) {
            $total += $item->getSubtotal();
        }

        $total += $this->getShippingTotalWithoutTax();
        $total += $this->getAdjustmentAmount(Adjustment::MANAGEMENT_FEES_ADJUSTMENT);

        return $total;
    }

    public function getManagementFeesTotal(): int
    {
        $total = $this->getAdjustmentsTotal(Adjustment::MANAGEMENT_FEES_ADJUSTMENT);
        $total += $this->getAdjustmentsTotal(Adjustment::MANAGEMENT_FEES_ADJUSTMENT_TAX);

        return $total;
    }

    /**
     * {@inheritdoc}
     */
    public function getTaxTotal(): int
    {
        $taxTotal = parent::getTaxTotal();

        foreach ($this->getAdjustments(Adjustment::MANAGEMENT_FEES_ADJUSTMENT_TAX) as $taxAdjustment) {
            $taxTotal += $taxAdjustment->getAmount();
        }

        return $taxTotal;
    }

    public function hasBlueCardPayment(): bool
    {
        return in_array($this->getLastPaymentMethodCode(), [Payment::PAYMENT_CITELIS, Payment::PAYMENT_CITELIS_PAYMENT_N]);
    }

    public function getEmailLinkClicked(): bool
    {
        return $this->emailLinkClicked;
    }

    public function setEmailLinkClicked(bool $emailLinkClicked): self
    {
        $this->emailLinkClicked = $emailLinkClicked;

        return $this;
    }
}
