<?php

declare(strict_types=1);

namespace App\Importer;

use App\Entity\Product;
use App\Entity\Supplier;
use App\Service\ProductService;
use Doctrine\ORM\EntityManagerInterface;
use Sylius\Component\Core\Model\Channel;
use Sylius\Component\Core\Model\ChannelPricing;
use Sylius\Component\Core\Model\ProductTaxon;
use Sylius\Component\Core\Model\ProductVariant;
use Sylius\Component\Core\Uploader\ImageUploaderInterface;
use Sylius\Component\Product\Generator\SlugGeneratorInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Sylius\Component\Shipping\Model\ShippingCategory;
use Sylius\Component\Taxation\Model\TaxCategory;
use Sylius\Component\Taxonomy\Model\Taxon;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * Class ArticleImporter.
 *
 * Query: SELECT `reference_ths`, `reference_ths_sage`, `reference_ths_affichage`,
`reference_ths_url`, f.libelle, `designation_courte`, `designation_detaillee`,
`specificite`, `dimensions_a`, `dimensions_b`, `matiere`, `pression_maximum`,
`temperature_maximum`, `diametre_nominal`, `reference_metrique_entree`,
`sexe_entree`, `precision_supplementaire_entree`, `reference_metrique_sortie`,
`sexe_sortie`, `precision_supplementaire_sortie`, `calibre_piece`, `debit_max`,
`hors_gabarit`, `piece_origine`, `piece_visible`, `poids`, `prix_vente_ht`,
`taux_remise_fournisseur`, `prix_vente_ths`, `marge_pourcentage`, `marge_euro`,
`prix_ttc`, `stock`, `url_article`, `url_image`, `balise_title`, `balise_meta`,
`minimum_commande`, fo.code, `statut_vente`
FROM `article` a
INNER JOIN `famille` f ON a.id_famille = f.id_famille
LEFT JOIN `fournisseur` fo ON a.id_fournisseur = fo.id
 */
class ArticleImporter implements ImporterInterface
{
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    /**
     * @var FactoryInterface
     */
    private $attributeValueFactory;
    /**
     * @var FactoryInterface
     */
    private $channelPricingFactory;

    /**
     * @var FactoryInterface
     */
    private $productFactory;

    /**
     * @var FactoryInterface
     */
    private $productImageFactory;

    /**
     * @var FactoryInterface
     */
    private $productVariantFactory;

    /**
     * @var RepositoryInterface
     */
    private $productTaxonFactory;

    /**
     * @var RepositoryInterface
     */
    private $channelRepository;

    /**
     * @var RepositoryInterface
     */
    private $productAttributeRepository;

    /**
     * @var RepositoryInterface
     */
    private $shippingCategoryRepository;

    /**
     * @var RepositoryInterface
     */
    private $supplierRepository;

    /**
     * @var RepositoryInterface
     */
    private $taxCategoryRepository;

    /**
     * @var RepositoryInterface
     */
    private $taxonRepository;

    /**
     * @var SlugGeneratorInterface
     */
    private $slugGenerator;

    /**
     * @var ImageUploaderInterface
     */
    private $imageUploader;

    /**
     * @var ProductService
     */
    private $productService;

    /**
     * ArticleImporter constructor.
     *
     * @param EntityManagerInterface $entityManager
     * @param FactoryInterface       $channelPricingFactory
     * @param FactoryInterface       $attributeValueFactory
     * @param FactoryInterface       $productFactory
     * @param FactoryInterface       $productImageFactory
     * @param FactoryInterface       $productTaxonFactory
     * @param FactoryInterface       $productVariantFactory
     * @param RepositoryInterface    $channelRepository
     * @param RepositoryInterface    $productAttributeRepository
     * @param RepositoryInterface    $shippingCategoryRepository
     * @param RepositoryInterface    $supplierRepository
     * @param RepositoryInterface    $taxCategoryRepository
     * @param RepositoryInterface    $taxonRepository
     * @param SlugGeneratorInterface $slugGenerator
     * @param ImageUploaderInterface $imageUploader
     * @param ProductService         $productService
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        FactoryInterface $channelPricingFactory,
        FactoryInterface $attributeValueFactory,
        FactoryInterface $productFactory,
        FactoryInterface $productImageFactory,
        FactoryInterface $productTaxonFactory,
        FactoryInterface $productVariantFactory,
        RepositoryInterface $channelRepository,
        RepositoryInterface $productAttributeRepository,
        RepositoryInterface $shippingCategoryRepository,
        RepositoryInterface $supplierRepository,
        RepositoryInterface $taxCategoryRepository,
        RepositoryInterface $taxonRepository,
        SlugGeneratorInterface $slugGenerator,
        ImageUploaderInterface $imageUploader,
        ProductService $productService
    ) {
        $this->entityManager = $entityManager;
        $this->channelPricingFactory = $channelPricingFactory;
        $this->attributeValueFactory = $attributeValueFactory;
        $this->productFactory = $productFactory;
        $this->productImageFactory = $productImageFactory;
        $this->productTaxonFactory = $productTaxonFactory;
        $this->productVariantFactory = $productVariantFactory;
        $this->shippingCategoryRepository = $shippingCategoryRepository;
        $this->supplierRepository = $supplierRepository;
        $this->channelRepository = $channelRepository;
        $this->productAttributeRepository = $productAttributeRepository;
        $this->taxCategoryRepository = $taxCategoryRepository;
        $this->taxonRepository = $taxonRepository;
        $this->slugGenerator = $slugGenerator;
        $this->imageUploader = $imageUploader;
        $this->productService = $productService;
    }

    /**
     * Import article from THS V1.
     *
     * @param array $line
     */
    public function import(array $line): void
    {
        $code = strtoupper($this->slugGenerator->generate($line[0]));
        $product = $this->entityManager->getRepository(Product::class)->findOneBy(['code' => $code]);

        if ($product) {
            $product->clearAttributes();
            /**
             * Attributes.
             */
            $product = $this->productService->addAttributesToProduct($product, 'specificity', '' != $line[7] ? $line[7] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'dimension_a', '' != $line[8] ? $line[8] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'dimension_b', '' != $line[9] ? $line[9] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'material', '' != $line[10] ? $line[10] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'maximum_pressure', '' != $line[11] ? $line[11] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'maximum_temperature', '' != $line[12] ? $line[12] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'nominal_diameter', '' != $line[13] ? $line[13] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'metric_reference_input', '' != $line[14] ? $line[14] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'sex_input', '' != $line[15] ? $line[15] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'additional_precision_input', '' != $line[16] ? $line[16] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'metric_reference_output', '' != $line[17] ? $line[17] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'sex_output', '' != $line[18] ? $line[18] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'additional_precision_output', '' != $line[19] ? $line[19] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'caliber_piece', '' != $line[20] ? $line[20] : '0');
            $product = $this->productService->addAttributesToProduct($product, 'maximum_flow', '' != $line[21] ? $line[21] : '0');

            return;
        }

        /**
         * Taxons.
         */
        $taxon = $this->getTaxon($line[4]);

        if (!$taxon) {
            return;
        }

        /** @var Product $product */
        $product = $this->productFactory->createNew();
        $product->setName($line[5]);
        $product->setCode($code);
        $product->setManufacturerReference($line[2]);
        $product->setMetaDescription($line[36]);
        $product->setEnabled('1' === $line[39]);
        $product->addChannel($this->getChannel());
        $product->setSlug($this->slugGenerator->generate(mb_strtolower($line[4].' '.$line[3])));
        $product->setShortDescription($line[5]);
        $product->setDescription($line[6]);
        $product->setMinimumOrder('1' == $line[37]);
        $product->setThsReferenceSage($line[1]);
        $product->setThsReferenceUrl($line[3]);
        $product->setNoIndex(isset($line[56]) ? '1' == $line[56] : false);

        $supplier = $this->getSupplier($line[38]);

        if ($supplier) {
            $product->setSupplier($supplier);
        }

        $product->setMainTaxon($taxon);

        /** @var ProductTaxon $productTaxon */
        $productTaxon = $this->productTaxonFactory->createNew();
        $productTaxon->setTaxon($taxon);
        $productTaxon->setProduct($product);
        $product->addProductTaxon($productTaxon);

        $taxon = $this->getTaxon('ARTICLE');

        /** @var ProductTaxon $productTaxon */
        $productTaxon = $this->productTaxonFactory->createNew();
        $productTaxon->setTaxon($taxon);
        $productTaxon->setProduct($product);
        $product->addProductTaxon($productTaxon);

        /**
         * Attributes.
         */
        $product = $this->productService->addAttributesToProduct($product, 'specificity', $line[7]);
        $product = $this->productService->addAttributesToProduct($product, 'dimension_a', $line[8]);
        $product = $this->productService->addAttributesToProduct($product, 'dimension_b', $line[9]);
        $product = $this->productService->addAttributesToProduct($product, 'material', $line[10]);
        $product = $this->productService->addAttributesToProduct($product, 'maximum_pressure', $line[11]);
        $product = $this->productService->addAttributesToProduct($product, 'maximum_temperature', $line[12]);
        $product = $this->productService->addAttributesToProduct($product, 'nominal_diameter', $line[13]);
        $product = $this->productService->addAttributesToProduct($product, 'metric_reference_input', $line[14]);
        $product = $this->productService->addAttributesToProduct($product, 'sex_input', $line[15]);
        $product = $this->productService->addAttributesToProduct($product, 'additional_precision_input', $line[16]);
        $product = $this->productService->addAttributesToProduct($product, 'metric_reference_output', $line[17]);
        $product = $this->productService->addAttributesToProduct($product, 'sex_output', $line[18]);
        $product = $this->productService->addAttributesToProduct($product, 'additional_precision_output', $line[19]);
        $product = $this->productService->addAttributesToProduct($product, 'caliber_piece', $line[20]);
        $product = $this->productService->addAttributesToProduct($product, 'maximum_flow', $line[21]);

        /** @var ProductVariant $productVariant */
        $productVariant = $this->productVariantFactory->createNew();

        $productVariant->setName($line[4]);
        $productVariant->setCode($line[0]);
        $productVariant->setPosition(1);
        $productVariant->setProduct($product);
        $productVariant->setWeight((float) $line[25]);
        $productVariant->setTaxCategory($this->getTaxCategory());
        $productVariant->setShippingCategory($this->getShippingCategory('1' == $line[22] ? 'OUTSIZE' : 'REGULAR'));

        if ('NULL' !== $line[28]) {
            /** @var ChannelPricing $channelPricing */
            $channelPricing = $this->channelPricingFactory->createNew();
            $channelPricing->setChannelCode($this->getChannel()->getCode());
            $channelPricing->setProductVariant($productVariant);
            $channelPricing->setPrice((int) ($line[28] * 100));
            $channelPricing->setOriginalPrice((int) ($line[28] * 100));

            $productVariant->addChannelPricing($channelPricing);
        }

        $product->addVariant($productVariant);

        $imagePath = __DIR__."/../../../old_images/articles/{$line[34]}.jpg";

        if ('' != $line[34] && file_exists($imagePath)) {
            $uploadedImage = new UploadedFile($imagePath, basename($imagePath));

            /** @var ImageInterface $productImage */
            $productImage = $this->productImageFactory->createNew();
            $productImage->setFile($uploadedImage);
            $productImage->setType($line[0]);
            $this->imageUploader->upload($productImage);
            $product->addImage($productImage);
        }

        $product->resetCommercialDocumentations();
        $product->resetTechnicalBursts();
        $product->resetTechnicalDocumentations();

        $this->entityManager->persist($product);
    }

    /**
     * @param string $code
     *
     * @return Taxon|null
     */
    private function getTaxon(string $code): ?Taxon
    {
        return $this->taxonRepository->findOneBy(['code' => $code]);
    }

    /**
     * @param string $code
     *
     * @return Supplier|null
     */
    private function getSupplier(string $code): ?Supplier
    {
        return $this->supplierRepository->findOneBy(['code' => $code]);
    }

    /**
     * @return Channel
     */
    private function getChannel(): Channel
    {
        return $this->channelRepository->findOneBy(['code' => 'ECOMMERCE']);
    }

    /**
     * @return TaxCategory
     */
    private function getTaxCategory(): TaxCategory
    {
        return $this->taxCategoryRepository->findOneBy(['code' => 'SALE20']);
    }

    /**
     * @param string $code
     *
     * @return ShippingCategory
     */
    private function getShippingCategory(string $code): ShippingCategory
    {
        return $this->shippingCategoryRepository->findOneBy(['code' => $code]);
    }
}
