<?php

declare(strict_types=1);

namespace App\Importer;

use Doctrine\ORM\EntityManagerInterface;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Product\Generator\SlugGeneratorInterface;
use Sylius\Component\Product\Model\ProductAssociation;
use Sylius\Component\Product\Model\ProductAssociationTypeInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;

/**
 * Class AssociationImporter.
 *
 *          SELECT nhp.modele, nhp.reference_ths, aaa.reference_ths
FROM `association` a
inner join association_article aa on a.`id_association` = aa.`id_association`
inner join article aaa on aaa.reference_ths_sage = aa.reference_ths_sage
inner join nettoyeur_haute_pression nhp on nhp.id_nettoyeur = a.id_nettoyeur
order by a.`id_nettoyeur`, aaa.reference_ths, a.ordre, aa.ordre
 */
class AssociationImporter implements ImporterInterface
{
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    /**
     * @var FactoryInterface
     */
    private $productAssociationFactory;

    /**
     * @var RepositoryInterface
     */
    private $productAssociationRepository;

    /**
     * @var RepositoryInterface
     */
    private $productAssociationTypeRepository;

    /**
     * @var RepositoryInterface
     */
    private $productRepository;

    /**
     * @var SlugGeneratorInterface
     */
    private $slugGenerator;

    /**
     * AssociationImporter constructor.
     *
     * @param EntityManagerInterface $entityManager
     * @param FactoryInterface       $productAssociationFactory
     * @param RepositoryInterface    $productAssociationRepository
     * @param RepositoryInterface    $productAssociationTypeRepository
     * @param RepositoryInterface    $productRepository
     * @param SlugGeneratorInterface $slugGenerator
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        FactoryInterface $productAssociationFactory,
        RepositoryInterface $productAssociationRepository,
        RepositoryInterface $productAssociationTypeRepository,
        RepositoryInterface $productRepository,
        SlugGeneratorInterface $slugGenerator
    ) {
        $this->entityManager = $entityManager;
        $this->productAssociationFactory = $productAssociationFactory;
        $this->productAssociationRepository = $productAssociationRepository;
        $this->productAssociationTypeRepository = $productAssociationTypeRepository;
        $this->productRepository = $productRepository;
        $this->slugGenerator = $slugGenerator;
    }

    /**
     * Import association from THS V1.
     *
     * @param array $line
     */
    public function import(array $line): void
    {
        $productToAssociate = $this->getProduct($this->slugGenerator->generate($line[2]));
        $product = $this->getProduct($this->slugGenerator->generate($line[0].' '.$line[1]));

        // Article
        $productAssociationType = $this->getProductAssociationType('ACCESSORIES');
        $this->addAssociation($productAssociationType, $product, $productToAssociate);

        // NHP
        $productAssociationType = $this->getProductAssociationType('NHP');
        $this->addAssociation($productAssociationType, $productToAssociate, $product);
    }

    /**
     * @param ProductAssociationTypeInterface $productAssociationType
     * @param ProductInterface                $product
     * @param ProductInterface                $productToAssociate
     */
    private function addAssociation(
        ProductAssociationTypeInterface $productAssociationType,
        ProductInterface $product,
        ProductInterface $productToAssociate
    ): void {
        $associations = $product->getAssociations();

        if (0 === $associations->count()) {
            /** @var ProductAssociation $productAssociation */
            $productAssociation = $this->productAssociationFactory->createNew();
            $productAssociation->setType($productAssociationType);
            $productAssociation->addAssociatedProduct($productToAssociate);
        } else {
            $productAssociations = $associations->filter(function (ProductAssociation $pa) use ($productAssociationType) {
                return $pa->getType()->getCode() === $productAssociationType->getCode();
            });

            if (0 === $productAssociations->count()) {
                /** @var ProductAssociation $productAssociation */
                $productAssociation = $this->productAssociationFactory->createNew();
                $productAssociation->setType($productAssociationType);
                $productAssociation->addAssociatedProduct($productToAssociate);
            } else {
                $productAssociation = $productAssociations->first();
                $product->removeAssociation($productAssociation);
                $productAssociation->addAssociatedProduct($productToAssociate);
            }
        }

        $product->addAssociation($productAssociation);
    }

    /**
     * @param string $code
     *
     * @return ProductAssociationTypeInterface
     */
    private function getProductAssociationType(string $code): ProductAssociationTypeInterface
    {
        return $this->productAssociationTypeRepository->findOneBy(['code' => $code]);
    }

    /**
     * @param string $code
     *
     * @return ProductInterface
     */
    private function getProduct(string $code): ProductInterface
    {
        return $this->productRepository->findOneBy(['code' => $code]);
    }
}
