<?php

declare(strict_types=1);

namespace App\Importer;

use App\Entity\Address;
use App\Entity\Adjustment;
use App\Entity\Customer;
use App\Entity\Order;
use App\Entity\Payment;
use Doctrine\ORM\EntityManagerInterface;
use SM\Factory\Factory;
use Sylius\Component\Core\Model\AdjustmentInterface;
use Sylius\Component\Core\Model\Channel;
use Sylius\Component\Core\Model\OrderItem;
use Sylius\Component\Core\Model\OrderItemUnit;
use Sylius\Component\Core\Model\PaymentMethod;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Core\Model\Shipment;
use Sylius\Component\Core\Model\ShippingMethod;
use Sylius\Component\Core\OrderPaymentTransitions;
use Sylius\Component\Core\OrderShippingTransitions;
use Sylius\Component\Order\OrderTransitions;
use Sylius\Component\Payment\PaymentTransitions;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Sylius\Component\Shipping\ShipmentTransitions;

/**
 * Class OrderImporter.
 */
class OrderImporter implements ImporterInterface
{
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    /**
     * @var FactoryInterface
     */
    private $addressFactory;

    /**
     * @var FactoryInterface
     */
    private $adjustmentFactory;

    /**
     * @var FactoryInterface
     */
    private $orderFactory;

    /**
     * @var FactoryInterface
     */
    private $orderItemFactory;

    /**
     * @var FactoryInterface
     */
    private $orderItemUnitFactory;

    /**
     * @var FactoryInterface
     */
    private $paymentFactory;

    /**
     * @var Factory
     */
    private $stateMachineFactory;

    /**
     * @var FactoryInterface
     */
    private $shipmentFactory;

    /**
     * @var RepositoryInterface
     */
    private $channelRepository;

    /**
     * @var RepositoryInterface
     */
    private $customerRepository;

    /**
     * @var RepositoryInterface
     */
    private $paymentMethodRepository;

    /**
     * @var RepositoryInterface
     */
    private $shippingMethodRepository;

    /** @var RepositoryInterface */
    private $productRepository;

    /**
     * OrderImporter constructor.
     *
     * @param EntityManagerInterface $entityManager
     * @param FactoryInterface       $addressFactory
     * @param FactoryInterface       $adjustmentFactory
     * @param FactoryInterface       $orderFactory
     * @param FactoryInterface       $orderItemFactory
     * @param FactoryInterface       $orderItemUnitFactory
     * @param FactoryInterface       $paymentFactory
     * @param Factory                $stateMachineFactory
     * @param FactoryInterface       $shipmentFactory
     * @param RepositoryInterface    $channelRepository
     * @param RepositoryInterface    $customerRepository
     * @param RepositoryInterface    $paymentMethodRepository
     * @param RepositoryInterface    $shippingMethodRepository
     * @param RepositoryInterface    $productRepository
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        FactoryInterface $addressFactory,
        FactoryInterface $adjustmentFactory,
        FactoryInterface $orderFactory,
        FactoryInterface $orderItemFactory,
        FactoryInterface $orderItemUnitFactory,
        FactoryInterface $paymentFactory,
        Factory $stateMachineFactory,
        FactoryInterface $shipmentFactory,
        RepositoryInterface $channelRepository,
        RepositoryInterface $customerRepository,
        RepositoryInterface $paymentMethodRepository,
        RepositoryInterface $shippingMethodRepository,
        RepositoryInterface $productRepository
    ) {
        $this->entityManager = $entityManager;
        $this->addressFactory = $addressFactory;
        $this->adjustmentFactory = $adjustmentFactory;
        $this->orderFactory = $orderFactory;
        $this->orderItemFactory = $orderItemFactory;
        $this->orderItemUnitFactory = $orderItemUnitFactory;
        $this->paymentFactory = $paymentFactory;
        $this->stateMachineFactory = $stateMachineFactory;
        $this->shipmentFactory = $shipmentFactory;
        $this->channelRepository = $channelRepository;
        $this->customerRepository = $customerRepository;
        $this->paymentMethodRepository = $paymentMethodRepository;
        $this->shippingMethodRepository = $shippingMethodRepository;
        $this->productRepository = $productRepository;
    }

    /**
     * Import order from THS V1.
     *
     * @param array $line
     *
     * @throws \SM\SMException
     */
    public function import(array $line): void
    {
        /*
         * CustomerId not exist
         */
        if ('NULL' == $line[34]) {
            $this->createForTicketOrQuotationWithUnknownCustomer($line);

            return;
        }

        /** @var Order $order */
        $order = $this->orderFactory->createNew();
        /** @var Payment $payment */
        $payment = $this->paymentFactory->createNew();
        /** @var Shipment $shipment */
        $shipment = $this->shipmentFactory->createNew();

        $createdAt = \DateTime::createFromFormat('Y-m-d H:i:s', $line[36]);
        $updatedAt = \DateTime::createFromFormat('Y-m-d H:i:s', $line[37]);
        $customer = $this->getCustomer((int) $line[34]);
        /** @var Address $address */
        $address = $customer->getAddresses()->first();
        /** @var Address $billingAddress */
        $billingAddress = $this->addressFactory->createNew();
        $billingAddress->setType($line[9] ?? $address->getType());
        $billingAddress->setCivility($line[10] ?? $address->getCivility());
        $billingAddress->setLastName($line[11] ?? $address->getLastName());
        $billingAddress->setFirstName($line[12] ?? $address->getFirstName());
        $billingAddress->setCompany($line[38] ?? $address->getCompany());
        $billingAddress->setFaxNumber($line[19] ?? $address->getFaxNumber());
        $billingAddress->setPhoneNumber($line[18] ?? $address->getPhoneNumber());
        $billingAddress->setCreatedAt($createdAt);
        $billingAddress->setUpdatedAt($updatedAt);
        $billingAddress->setStreet($line[13].('' != $line[14] ? ' '.$line[14] : '') ?? $address->getStreet());
        $billingAddress->setCity($line[16] ?? $address->getCity());
        $billingAddress->setPostcode($line[15] ?? $address->getPostcode());
        $billingAddress->setCountryCode('FR');
        $order->setBillingAddress($billingAddress);

        /** @var Address $shippingAddress */
        $shippingAddress = $this->addressFactory->createNew();
        $shippingAddress->setType($line[9] ?? $billingAddress->getType());
        $shippingAddress->setCivility($line[10] ?? $billingAddress->getCivility());
        $shippingAddress->setLastName($line[11] ?? $billingAddress->getLastName());
        $shippingAddress->setFirstName($line[12] ?? $billingAddress->getFirstName());
        $shippingAddress->setCompany($line[38] ?? $billingAddress->getCompany());
        $shippingAddress->setFaxNumber($line[19] ?? $billingAddress->getFaxNumber());
        $shippingAddress->setPhoneNumber($line[18] ?? $billingAddress->getPhoneNumber());
        $shippingAddress->setCreatedAt($createdAt);
        $shippingAddress->setUpdatedAt($updatedAt);
        $shippingAddress->setStreet($line[13].('' != $line[14] ? ' '.$line[14] : '') ?? $billingAddress->getStreet());
        $shippingAddress->setCity($line[16] ?? $billingAddress->getCity());
        $shippingAddress->setPostcode($line[15] ?? $billingAddress->getPostcode());
        // set to FR => $line[17] ?? $billingAddress->getCountryCode()
        $shippingAddress->setCountryCode('FR');
        $order->setShippingAddress($shippingAddress);

        $order->setChannel($this->getChannel());
        $order->setCustomer($customer);
        $order->setNotes($line[21]);
        $order->setCheckoutCompletedAt($createdAt);
        $order->setCurrencyCode('EUR');
        $order->setLocaleCode('FR');
        $order->setCreatedAt($createdAt);
        $order->setUpdatedAt($updatedAt);
        $order->setType((int) $line[22]);
        $order->setOldOrderId((int) $line[0]);
        $order->setWeight((float) $line[1]);
        $order->setOrderToken('' != $line[23] ? $line[23] : $line[24]);
        $order->setQuotationToken($line[24]);
        $order->setQuotationPdf($line[25]);
        $order->setMailOrder('1' == $line[32]);
        $order->setMailColissimo('1' == $line[33]);
        $order->setIdUserTracking((int) $line[35]);
        $order = $this->addTicketQuotationOrderItem($order, $line);

        // Shipping
        if ($line[2] > 0) {
            /** @var Adjustment $adjustment */
            $adjustment = $this->adjustmentFactory->createNew();
            $adjustment->setType(AdjustmentInterface::SHIPPING_ADJUSTMENT);
            $amount = explode('.', (string) ($line[2] * 100));
            $adjustment->setAmount((int) $amount[0]);

            $order->addAdjustment($adjustment);

            // Shipping tax
            /** @var Adjustment $adjustment */
            $adjustment = $this->adjustmentFactory->createNew();
            // Tax
            $label = '20.00' == $line[6]
                ? 'Taux normal 20%'
                : 'Taux normal 19,6%';

            $vatRate = '20.00' == $line[6]
                ? 1.2
                : 1.196;

            $adjustment->setType(AdjustmentInterface::TAX_ADJUSTMENT);
            $adjustment->setLabel($label);
            $amount = explode('.', (string) (round($line[2] * $vatRate - $line[2], 2) * 100));
            $adjustment->setAmount((int) $amount[0]);
            $adjustment->setNeutral(false);

            $order->addAdjustment($adjustment);
        }

        // Some order in V1 don't have management fees but in real they must have, maybe a bug ?
        $managementFees = 0;

        if ('NULL' != $line[2] && 'NULL' != $line[4] && 'NULL' != $line[5]) {
            $managementFees = $line[5] - $line[2] - $line[4];
        }

        // Management fees
        if ($line[3] > 0) {
            /** @var Adjustment $adjustment */
            $adjustment = $this->adjustmentFactory->createNew();
            $adjustment->setType(Adjustment::MANAGEMENT_FEES_ADJUSTMENT);
            $amount = explode('.', (string) ($line[3] * 100));
            $adjustment->setAmount((int) $amount[0]);

            $order->addAdjustment($adjustment);

            // Management fees tax
            /** @var Adjustment $adjustment */
            $adjustment = $this->adjustmentFactory->createNew();
            // Tax
            $label = '20.00' == $line[6]
                ? 'Taux normal 20%'
                : 'Taux normal 19,6%';

            $vatRate = '20.00' == $line[6]
                ? 1.2
                : 1.196;

            $adjustment->setType(Adjustment::MANAGEMENT_FEES_ADJUSTMENT_TAX);
            $adjustment->setLabel($label);
            $amount = explode('.', (string) (round($line[3] * $vatRate - $line[3], 2) * 100));
            $adjustment->setAmount((int) $amount[0]);
            $adjustment->setNeutral(false);

            $order->addAdjustment($adjustment);
        }

        if (0 == $line[3] && $managementFees > 0) {
            /** @var Adjustment $adjustment */
            $adjustment = $this->adjustmentFactory->createNew();
            $adjustment->setType(Adjustment::MANAGEMENT_FEES_ADJUSTMENT);
            $amount = explode('.', (string) ($managementFees * 100));
            $adjustment->setAmount((int) $amount[0]);

            $order->addAdjustment($adjustment);

            // Management fees tax
            /** @var Adjustment $adjustment */
            $adjustment = $this->adjustmentFactory->createNew();
            // Tax
            $label = '20.00' == $line[6]
                ? 'Taux normal 20%'
                : 'Taux normal 19,6%';

            $vatRate = '20.00' == $line[6]
                ? 1.2
                : 1.196;

            $adjustment->setType(Adjustment::MANAGEMENT_FEES_ADJUSTMENT_TAX);
            $adjustment->setLabel($label);
            $amount = explode('.', (string) (round($managementFees * $vatRate - $managementFees, 2) * 100));
            $adjustment->setAmount((int) $amount[0]);
            $adjustment->setNeutral(false);

            $order->addAdjustment($adjustment);
        }

        // Payment
        $payment->setMethod($this->getPaymentMethod($line[26] > 0 ? (int) $line[26] : 1));
        $payment->setCurrencyCode('EUR');
        $line[7] = 'NULL' == $line[7] ? 0 : $line[7];
        $amount = explode('.', (string) ($line[7] * 100));
        $payment->setAmount((int) $amount[0]);
        $payment->setCreatedAt($createdAt);
        $payment->setUpdatedAt($updatedAt);

        if (1 == $line[26]) {
            $payment->setTransactionNumber($line[27]);
            $payment->setCardNumber($line[28]);
            $payment->setTransactionCertificat($line[29]);
            $payment->setTransactionDate(\DateTime::createFromFormat('Y-m-d H:i:s', $line[30]));
        }

        $order->addPayment($payment);

        $stateMachine = $this->stateMachineFactory->get($order, OrderPaymentTransitions::GRAPH);
        $stateMachine->apply(OrderPaymentTransitions::TRANSITION_REQUEST_PAYMENT);

        $stateMachine = $this->stateMachineFactory->get($payment, PaymentTransitions::GRAPH);
        $stateMachine->apply(PaymentTransitions::TRANSITION_CREATE);

        if (2 == $line[8]) {
            $stateMachine->apply(PaymentTransitions::TRANSITION_CANCEL);

            $stateMachine = $this->stateMachineFactory->get($order, OrderPaymentTransitions::GRAPH);
            $stateMachine->apply(OrderPaymentTransitions::TRANSITION_CANCEL);
        } elseif (in_array($line[8], [3, 7, 8, 4, 9])) {
            $stateMachine->apply(PaymentTransitions::TRANSITION_COMPLETE);
        }

        // Shipment
        $shipment->setMethod($this->getShippingMethod());
        $shipment->setTracking($line[31]);
        $shipment->setCreatedAt($createdAt);
        $shipment->setUpdatedAt($updatedAt);

        $order->addShipment($shipment);

        $stateMachine = $this->stateMachineFactory->get($order, OrderShippingTransitions::GRAPH);
        $stateMachine->apply(OrderShippingTransitions::TRANSITION_REQUEST_SHIPPING);

        $stateMachine = $this->stateMachineFactory->get($shipment, ShipmentTransitions::GRAPH);
        $stateMachine->apply(ShipmentTransitions::TRANSITION_CREATE);

        switch ($line[8]) {
            case 4: // Expédition colissimo
                $stateMachine->apply('process');
                $stateMachine->apply(ShipmentTransitions::TRANSITION_SHIP);
                break;
            case 7: // Commande traitée
                $stateMachine->apply('process');
                break;
            case 8: // Reliquat
                $stateMachine->apply('process');
                $stateMachine->apply('remainder');
                break;
            case 9: // Expédition par transporteur
                $stateMachine->apply('process');
                $stateMachine->apply('ship_outsize');
                break;
        }

        $stateMachine = $this->stateMachineFactory->get($order, OrderTransitions::GRAPH);
        $stateMachine->apply(OrderTransitions::TRANSITION_CREATE);

        switch ($line[8]) {
            case 5: // Annulée
                $stateMachine->apply(OrderTransitions::TRANSITION_CANCEL);
                break;
            case 4: // Expédition colissimo
            case 9: // Expédition par transporteur
                $stateMachine->apply(OrderTransitions::TRANSITION_FULFILL);
                break;
        }

        $this->entityManager->persist($order);
        $this->entityManager->flush();
    }

    private function createForTicketOrQuotationWithUnknownCustomer($line): void
    {
        $createdAt = \DateTime::createFromFormat('Y-m-d H:i:s', $line['36']);
        $updatedAt = \DateTime::createFromFormat('Y-m-d H:i:s', $line['37']);

        /** @var Order $order */
        $order = $this->orderFactory->createNew();
        $order->setChannel($this->getChannel());
        $order->setNotes($line[21]);
        $order->setCheckoutCompletedAt($createdAt);
        $order->setCurrencyCode('EUR');
        $order->setLocaleCode('FR');
        $order->setCreatedAt($createdAt);
        $order->setUpdatedAt($updatedAt);
        $order->setType((int) $line[22]);
        $order->setOldOrderId((int) $line[0]);
        $order->setWeight((float) $line[1]);
        $order->setOrderToken('' != $line[24] ? $line[24] : $line[23]);
        $order->setQuotationToken($line[24]);
        $order->setQuotationPdf($line[25]);
        $order->setMailOrder('1' == $line[32]);
        $order->setMailColissimo('1' == $line[33]);
        $order->setIdUserTracking((int) $line[35]);
        $order = $this->addTicketQuotationOrderItem($order, $line);

        $customer = $this->customerRepository->findOneBy(['email' => $line['20']]);

        if (!$customer) {
            /** @var Customer $customer */
            $customer = new Customer();
            $customer->setCompanyName('' != $line['38'] ? $line['38'] : 'unknown');
            $customer->setLastName('' != $line['11'] ? $line['11'] : 'unknown');
            $customer->setFirstName('' != $line['12'] ? $line['12'] : 'unknown');
            $customer->setEmail($line['20']);
            $customer->setPhoneNumber('' != $line['18'] ? $line['18'] : 'unknown');
            $customer->setCreatedAt($createdAt);
            $customer->setUpdatedAt($updatedAt);
        }

        $order->setCustomer($customer);

        /** @var \Sylius\Component\Core\Model\Address $billingAddress */
        $billingAddress = $this->addressFactory->createNew();
        $billingAddress->setLastName('' != $line['11'] ? $line['11'] : 'unknown');
        $billingAddress->setFirstName('' != $line['12'] ? $line['12'] : 'unknown');
        $billingAddress->setCompany('' != $line['38'] ? $line['38'] : 'unknown');
        $billingAddress->setPhoneNumber('' != $line['18'] ? $line['18'] : 'unknown');
        $billingAddress->setCreatedAt($createdAt);
        $billingAddress->setUpdatedAt($updatedAt);
        $billingAddress->setStreet('' != $line['13'] ? $line['13'] : 'unknown');
        $billingAddress->setCity('' != $line['16'] ? $line['16'] : 'unknown');
        $billingAddress->setPostcode('' != $line['15'] ? $line['15'] : 'unknown');
        $billingAddress->setCountryCode('FR');
        $order->setBillingAddress($billingAddress);

        /** @var Address $shippingAddress */
        $shippingAddress = $this->addressFactory->createNew();
        $shippingAddress->setLastName('' != $line['11'] ? $line['11'] : 'unknown');
        $shippingAddress->setFirstName('' != $line['12'] ? $line['12'] : 'unknown');
        $shippingAddress->setCompany('' != $line['38'] ? $line['38'] : 'unknown');
        $shippingAddress->setPhoneNumber('' != $line['18'] ? $line['18'] : 'unknown');
        $shippingAddress->setCreatedAt($createdAt);
        $shippingAddress->setUpdatedAt($updatedAt);
        $shippingAddress->setStreet('' != $line['13'] ? $line['13'] : 'unknown');
        $shippingAddress->setCity('' != $line['16'] ? $line['16'] : 'unknown');
        $shippingAddress->setPostcode('' != $line['15'] ? $line['15'] : 'unknown');
        $shippingAddress->setCountryCode('FR');
        $order->setShippingAddress($shippingAddress);

        $stateMachine = $this->stateMachineFactory->get($order, OrderTransitions::GRAPH);

        $stateMachine->apply(OrderTransitions::TRANSITION_CREATE);
        $stateMachine->apply('offer');
        $this->entityManager->persist($order);
        $this->entityManager->flush();
    }

    /**
     * @param Order $order
     * @param array $line
     *
     * @return Order
     */
    private function addTicketQuotationOrderItem(Order $order, array $line): Order
    {
        // Add product for ticket and quotation
        // 1 = quotation, 3 = ticket
        if (in_array($order->getType(), [1, 3])) {
            if (1 === $order->getType()) {
                $product = $this->getProduct('QUOTATION');
            } else {
                $product = $this->getProduct('TICKET');
            }

            $vatRate = $order->getCreatedAt() < \DateTime::createFromFormat('Y-m-d H:i:s', '2014-01-01 00:00:00') ? 19.6 : 20;
            $vat = $order->getCreatedAt() < \DateTime::createFromFormat('Y-m-d H:i:s', '2014-01-01 00:00:00') ? 1.196 : 1.2;
            $productVariant = $product->getVariants()->first();

            /** @var OrderItem $orderItem */
            $orderItem = $this->orderItemFactory->createNew();
            $orderItem->setVariant($productVariant);
            $priceWithVat = 'NULL' !== $line[7] ? $line[7] : 0;
            $priceWithoutVat = round($priceWithVat / $vat, 2);
            $priceVat = $priceWithVat - $priceWithoutVat;
            $priceWithoutVat = explode('.', (string) ($priceWithoutVat * 100));
            $orderItem->setUnitPrice((int) $priceWithoutVat[0]);
            $orderItem->setProductName($product->getName());
            $orderItem->setVariantName($productVariant->getName());

            /** @var OrderItemUnit $orderItemUnit */
            $orderItemUnit = $this->orderItemUnitFactory->createForItem($orderItem);
            $orderItemUnit->setCreatedAt($order->getCreatedAt());
            $orderItemUnit->setUpdatedAt($order->getUpdatedAt());

            // Tax
            /** @var Adjustment $adjustment */
            $adjustment = $this->adjustmentFactory->createNew();
            $adjustment->setType(AdjustmentInterface::TAX_ADJUSTMENT);
            $adjustment->setLabel('Taux normal '.$vatRate.'%');
            $price = explode('.', (string) ($priceVat * 100));
            $adjustment->setAmount((int) $price[0]);
            $adjustment->setNeutral(false);
            $orderItemUnit->addAdjustment($adjustment);

            $orderItem->addUnit($orderItemUnit);
            $order->addItem($orderItem);
        }

        return $order;
    }

    /**
     * @return Channel
     */
    private function getChannel(): Channel
    {
        return $this->channelRepository->findOneBy(['code' => 'ECOMMERCE']);
    }

    /**
     * @param int $id
     *
     * @return Customer
     */
    private function getCustomer(int $id): Customer
    {
        return $this->customerRepository->findOneBy(['oldCustomerId' => $id]);
    }

    /**
     * @param int $id
     *
     * @return PaymentMethod
     */
    private function getPaymentMethod(int $id): PaymentMethod
    {
        $codes = [
            1 => 'CITELIS',
            2 => 'CHQ',
            3 => 'MANDAT',
        ];

        return $this->paymentMethodRepository->findOneBy(['code' => $codes[$id]]);
    }

    /**
     * @return ShippingMethod
     */
    private function getShippingMethod(): ShippingMethod
    {
        return $this->shippingMethodRepository->findOneBy(['code' => 'FR-LP']);
    }

    /**
     * @param string $code
     *
     * @return ProductInterface|null
     */
    private function getProduct(string $code): ?ProductInterface
    {
        return $this->productRepository->findOneBy(['code' => $code]);
    }
}
