<?php

declare(strict_types=1);

namespace App\OrderProcessing;

use App\Entity\Setting;
use Sylius\Component\Core\Model\AdjustmentInterface;
use Sylius\Component\Order\Model\OrderInterface as BaseOrderInterface;
use Sylius\Component\Order\Processor\OrderProcessorInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;

final class OrderFreeShippingProcessor implements OrderProcessorInterface
{
    /** @var RepositoryInterface */
    private $repositorySetting;

    /** @var FactoryInterface */
    private $factoryAdjustment;

    /**
     * OrderManagementFeesProcessor constructor.
     *
     * @param RepositoryInterface $repositorySetting
     * @param FactoryInterface    $factoryAdjustment
     */
    public function __construct(RepositoryInterface $repositorySetting, FactoryInterface $factoryAdjustment)
    {
        $this->repositorySetting = $repositorySetting;
        $this->factoryAdjustment = $factoryAdjustment;
    }

    /**
     * {@inheritdoc}
     */
    public function process(BaseOrderInterface $order): void
    {
        /** @var Setting|null $shippingFeesSetting */
        $shippingFeesSetting = $this->repositorySetting->findOneBy([
            'code' => Setting::MINIMUM_SHIPPING_FEES,
            'active' => true,
        ]);

        $order->removeAdjustments(AdjustmentInterface::ORDER_SHIPPING_PROMOTION_ADJUSTMENT);

        if (
            $order->getShippingAddress()
            && $shippingFeesSetting
            && (int) $shippingFeesSetting->getContent() > 0
            && $order->getItemsTotal() >= (int) $shippingFeesSetting->getContent()
            && 'FR' === $order->getShippingAddress()->getCountryCode()
        ) {
            /** @var Adjustment $adjustment */
            $adjustment = $this->factoryAdjustment->createNew();
            $adjustment->setType(AdjustmentInterface::ORDER_SHIPPING_PROMOTION_ADJUSTMENT);
            $adjustment->setAmount($order->getShippingTotal() * -1);

            $order->addAdjustment($adjustment);
        }
    }
}
