<?php

declare(strict_types=1);

namespace App\Repository;

use App\Entity\Product;
use Doctrine\DBAL\ConnectionException;
use Exception;
use Sylius\Bundle\CoreBundle\Doctrine\ORM\ProductRepository as baseProductRepository;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Model\Taxon;

class ProductRepository extends baseProductRepository
{
    public function findLatestPromotionByChannel(ChannelInterface $channel, string $locale, int $count): array
    {
        return $this->createQueryBuilder('o')
            ->addSelect('translation')
            ->innerJoin('o.translations', 'translation', 'WITH', 'translation.locale = :locale')
            ->andWhere(':channel MEMBER OF o.channels')
            ->andWhere('o.enabled = true')
            ->andWhere('o.promotion = true')
            ->addOrderBy('o.createdAt', 'DESC')
            ->setParameter('channel', $channel)
            ->setParameter('locale', $locale)
            ->setMaxResults($count)
            ->getQuery()
            ->getResult();
    }

    public function findByCodes(array $codes): array
    {
        return $this->createQueryBuilder('p')
            ->innerJoin('p.translations', 'translation', 'WITH', 'translation.locale = :locale')
            ->andWhere('p.code IN (:codes)')
            ->setParameters([
                'codes' => $codes,
                'locale' => 'fr_FR',
            ])
            ->getQuery()
            ->getResult();
    }

    public function findByBooleanAttribute(
        Taxon $mainTaxon,
        string $attributeCode,
        bool $value = true
    ): array {
        return $this->createQueryBuilder('p')
            ->innerJoin('p.mainTaxon', 'm')
            ->innerJoin('p.attributes', 'at')
            ->innerJoin('at.attribute', 'a')
            ->innerJoin('p.translations', 'translation', 'WITH', 'translation.locale = :locale')
            ->andWhere('m = :mainTaxon')
            ->andWhere('at.boolean = :value')
            ->andWhere('a.code = :attributeCode')
            ->andWhere('p.enabled = true')
            ->setParameters([
                'mainTaxon' => $mainTaxon,
                'attributeCode' => $attributeCode,
                'value' => $value,
                'locale' => 'fr_FR',
            ])
            ->orderBy('translation.name', 'ASC')
            ->getQuery()
            ->getResult();
    }

    public function findAllBooleanAttribute(Taxon $mainTaxon)
    {
        return $this->createQueryBuilder('p')
            ->innerJoin('p.mainTaxon', 'm')
            ->innerJoin('p.attributes', 'at')
            ->innerJoin('at.attribute', 'a')
            ->innerJoin('p.translations', 'translation', 'WITH', 'translation.locale = :locale')
            ->andWhere('m = :mainTaxon')
            ->andWhere('at.boolean = true')
            ->andWhere('p.enabled = true')
            ->setParameters([
                'mainTaxon' => $mainTaxon,
                'locale' => 'fr_FR',
            ])
            ->orderBy('a.position', 'ASC')
            ->getQuery()
            ->getResult();
    }

    public function findByNothing(Taxon $mainTaxon): array
    {
        return $this->createQueryBuilder('p')
            ->innerJoin('p.mainTaxon', 'm')
            ->innerJoin('p.translations', 'translation', 'WITH', 'translation.locale = :locale')
            ->andWhere('m = :mainTaxon')
            ->andWhere('p.enabled = true')
            ->setParameters([
                'mainTaxon' => $mainTaxon,
                'locale' => 'fr_FR',
            ])
            ->orderBy('translation.name', 'ASC')
            ->getQuery()
            ->getResult();
    }

    /**
     * @return mixed
     */
    public function searchByReference(string $reference)
    {
        return $this->createQueryBuilder('p')
            ->andWhere('p.enabled = true')
            ->andWhere('p.onSale = true')
            ->andWhere('p.displayPrice = true')
            ->andWhere('p.noLongerManufactured = false')
            ->andWhere('p.code LIKE :reference OR p.thsReferenceSage LIKE :reference OR p.manufacturerReference LIKE :reference')
            ->addOrderBy('p.code')
            ->addOrderBy('p.thsReferenceSage')
            ->setParameter('reference', $reference.'%')
            ->setMaxResults(10)
            ->getQuery()
            ->getResult();
    }

    public function findSearchableProductByTaxon(int $taxonId)
    {
        return $this->createQueryBuilder('p')
            ->select('p', 't')
            ->innerJoin('p.translations', 't', 'WITH', 't.locale = :locale')
            ->andWhere('p.enabled = true')
            ->andWhere('p.displaySearch = true')
            ->andWhere('p.mainTaxon = :taxon')
            ->addOrderBy('t.name')
            ->setParameter('taxon', $taxonId)
            ->setParameter('locale', 'fr_FR')
            ->getQuery()
            ->getResult();
    }

    public function findAllIterate(): iterable
    {
        foreach ($this->createQueryBuilder('p')
            ->getQuery()
            ->iterate() as $product) {
            yield $product[0];
        }
    }

    /**
     * @return bool|string
     *
     * @throws ConnectionException
     */
    public function finalizeImport(Taxon $taxon)
    {
        $con = $this->getEntityManager()->getConnection();

        try {
            $con->beginTransaction();

            $result = $this->createQueryBuilder('p')
                ->leftJoin('p.associations', 'a')
                ->leftJoin('a.associatedProducts', 'ap')
                ->andWhere('a.id IS NULL')
                ->andWhere('ap.id IS NULL')
                ->andWhere('p.mainTaxon = :id')
                ->andWhere('p.imported = :imported')
                ->setParameters([
                    'id' => $taxon->getId(),
                    'imported' => false,
                ])
                ->getQuery()
                ->getResult();

            /** @var Product $product */
            foreach ($result as $product) {
                $product->deactivate();
            }

            $this->_em->flush();

            // Mark not imported all product of a taxon
            $this->getEntityManager()->createQueryBuilder()
                ->update(Product::class, 'p')
                ->set('p.imported', ':imported')
                ->andWhere('p.mainTaxon = :id')
                ->setParameters([
                    'id' => $taxon->getId(),
                    'imported' => false,
                ])
                ->getQuery()
                ->execute();

            $con->commit();

            return true;
        } catch (Exception $e) {
            $con->rollBack();

            return $e->getMessage();
        }
    }

    /**
     * @return mixed
     */
    public function findBySearch(string $term)
    {
        return $this->createQueryBuilder('o')
            ->select('CONCAT(o.code, \' - \', translation.name) AS name')
            ->addSelect('o.code')
            ->innerJoin('o.translations', 'translation')
            ->addOrderBy('o.updatedAt', 'DESC')
            ->andWhere('o.code LIKE :term')
            ->orWhere('translation.name LIKE :term')
            ->setParameter('term', '%'.$term.'%')
            ->setMaxResults(20)
            ->getQuery()
            ->getResult();
    }
}
