<?php

declare(strict_types=1);

namespace App\Service;

use App\Entity\Payment;
use App\Mailer\Emails as AppEmails;
use Sylius\Bundle\CoreBundle\Mailer\Emails;
use Sylius\Bundle\ShopBundle\EmailManager\OrderEmailManagerInterface;
use Sylius\Component\Core\Model\AdminUserInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\ShipmentInterface;
use Sylius\Component\Core\OrderPaymentStates;
use Sylius\Component\Mailer\Sender\SenderInterface;

class OrderEmailManager implements OrderEmailManagerInterface
{
    /** @var SenderInterface */
    private $emailSender;

    /** @var array */
    private $orderRecipients;

    /**
     * OrderEmailManager constructor.
     *
     * @param SenderInterface $emailSender
     * @param array           $orderRecipients
     */
    public function __construct(SenderInterface $emailSender, array $orderRecipients)
    {
        $this->emailSender = $emailSender;
        $this->orderRecipients = $orderRecipients;
    }

    /**
     * @param OrderInterface $order
     */
    public function sendConfirmationEmail(OrderInterface $order): void
    {
        if (!$order->hasBlueCardPayment()) {
            $this->emailSender->send(Emails::ORDER_CONFIRMATION, [$order->getCustomer()->getEmail()], ['order' => $order]);
        }
    }

    /**
     * @param OrderInterface $order
     */
    public function sendPaymentConfirmation(OrderInterface $order): void
    {
        /** @var Payment $payment */
        $payment = $order->getPayments()->last();
        $state = $payment->getOrder()->getPaymentState();

        if (OrderPaymentStates::STATE_PAID === $state) {
            if ($order->hasBlueCardPayment()) {
                $this->emailSender->send(Emails::ORDER_CONFIRMATION, [$order->getCustomer()->getEmail()], ['order' => $order]);
                $this->emailSender->send(AppEmails::ORDER_PAYMENT_DETAILS, $this->orderRecipients, ['order' => $order]);

                /** @var AdminUserInterface|null $salesman */
                if ($salesman = $order->getSalesman()) {
                    $this->emailSender->send(AppEmails::ORDER_PAYMENT_DETAILS, [$salesman->getEmail()], ['order' => $order]);
                }
            }

            $this->emailSender->send(AppEmails::ORDER_PAYMENT_SUCCEEDED, $this->orderRecipients, ['order' => $order]);

            /** @var AdminUserInterface|null $salesman */
            if ($salesman = $order->getSalesman()) {
                $this->emailSender->send(AppEmails::ORDER_PAYMENT_SUCCEEDED, [$salesman->getEmail()], ['order' => $order]);
            }
        } elseif (OrderPaymentStates::STATE_CANCELLED === $state) {
            $this->emailSender->send(AppEmails::ORDER_PAYMENT_FAILED, $this->orderRecipients, ['order' => $order]);

            /** @var AdminUserInterface|null $salesman */
            if ($salesman = $order->getSalesman()) {
                $this->emailSender->send(AppEmails::ORDER_PAYMENT_FAILED, [$salesman->getEmail()], ['order' => $order]);
            }
        }
    }

    /**
     * @param OrderInterface $order
     */
    public function sendDetails(OrderInterface $order): void
    {
        if (!$order->hasBlueCardPayment()) {
            $this->emailSender->send(AppEmails::ORDER_PAYMENT_DETAILS, $this->orderRecipients, ['order' => $order]);

            /** @var AdminUserInterface|null $salesman */
            if ($salesman = $order->getSalesman()) {
                $this->emailSender->send(AppEmails::ORDER_PAYMENT_DETAILS, [$salesman->getEmail()], ['order' => $order]);
            }
        }
    }

    /**
     * @param ShipmentInterface $shipment
     */
    public function sendRemainder(ShipmentInterface $shipment): void
    {
        $this->emailSender->send(AppEmails::ORDER_SHIPMENT_REMAINDER, [$shipment->getOrder()->getUser()->getEmail()]);
    }

    /**
     * @param ShipmentInterface $shipment
     */
    public function sendProcess(ShipmentInterface $shipment): void
    {
        $this->emailSender->send(AppEmails::ORDER_SHIPMENT_PROCESS, [$shipment->getOrder()->getUser()->getEmail()]);
    }

    /**
     * @param ShipmentInterface $shipment
     */
    public function sendShipOutsize(ShipmentInterface $shipment): void
    {
        $this->emailSender->send(AppEmails::ORDER_SHIPMENT_SHIP_OUTSIZE, [$shipment->getOrder()->getUser()->getEmail()]);
    }
}
