<?php

declare(strict_types=1);

namespace App\Service;

use App\Entity\Product;
use App\Entity\Supplier;
use Doctrine\ORM\EntityManagerInterface;
use Sylius\Component\Attribute\Model\AttributeValueInterface;
use Sylius\Component\Core\Model\Channel;
use Sylius\Component\Core\Model\ChannelPricing;
use Sylius\Component\Core\Model\ProductVariant;
use Sylius\Component\Core\Model\Taxon;
use Sylius\Component\Core\Uploader\ImageUploaderInterface;
use Sylius\Component\Product\Generator\SlugGeneratorInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\HttpFoundation\File\UploadedFile;

class ProductService
{
    /**
     * @var EntityManagerInterface
     */
    protected $entityManager;

    /**
     * @var FactoryInterface
     */
    protected $factoryAttributeValue;

    /**
     * @var FactoryInterface
     */
    protected $factoryChannelPricing;

    /**
     * @var FactoryInterface
     */
    protected $factoryProductImage;

    /**
     * @var FactoryInterface
     */
    protected $factoryProductTaxon;

    /**
     * @var FactoryInterface
     */
    protected $factoryProductVariant;

    /**
     * @var RepositoryInterface
     */
    protected $repositoryChannel;

    /**
     * @var RepositoryInterface
     */
    protected $repositoryProduct;

    /**
     * @var RepositoryInterface
     */
    protected $repositoryProductAttribute;

    /**
     * @var RepositoryInterface
     */
    protected $repositoryShippingCategory;

    /**
     * @var RepositoryInterface
     */
    protected $repositorySupplier;

    /**
     * @var RepositoryInterface
     */
    protected $repositorySupplierCoefficient;

    /**
     * @var RepositoryInterface
     */
    protected $repositoryTaxCategory;

    /**
     * @var RepositoryInterface
     */
    protected $repositoryTaxon;

    /**
     * @var SlugGeneratorInterface
     */
    protected $slugGenerator;

    /**
     * @var string
     */
    protected $kernelProjectDir;

    /**
     * @var ImageUploaderInterface
     */
    protected $imageUploader;

    /**
     * @var array
     */
    protected $appFtpImagesDir;

    /**
     * ProductService constructor.
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        FactoryInterface $factoryAttributeValue,
        FactoryInterface $factoryChannelPricing,
        FactoryInterface $factoryProductImage,
        FactoryInterface $factoryProductTaxon,
        FactoryInterface $factoryProductVariant,
        RepositoryInterface $repositoryChannel,
        RepositoryInterface $repositoryProduct,
        RepositoryInterface $repositoryProductAttribute,
        RepositoryInterface $repositoryShippingCategory,
        RepositoryInterface $repositorySupplier,
        RepositoryInterface $repositorySupplierCoefficient,
        RepositoryInterface $repositoryTaxCategory,
        RepositoryInterface $repositoryTaxon,
        SlugGeneratorInterface $slugGenerator,
        ImageUploaderInterface $imageUploader,
        string $kernelProjectDir,
        array $appFtpImagesDir
    ) {
        $this->entityManager = $entityManager;
        $this->factoryAttributeValue = $factoryAttributeValue;
        $this->factoryChannelPricing = $factoryChannelPricing;
        $this->factoryProductImage = $factoryProductImage;
        $this->factoryProductTaxon = $factoryProductTaxon;
        $this->factoryProductVariant = $factoryProductVariant;
        $this->repositoryChannel = $repositoryChannel;
        $this->repositoryProduct = $repositoryProduct;
        $this->repositoryProductAttribute = $repositoryProductAttribute;
        $this->repositoryShippingCategory = $repositoryShippingCategory;
        $this->repositorySupplier = $repositorySupplier;
        $this->repositorySupplierCoefficient = $repositorySupplierCoefficient;
        $this->repositoryTaxCategory = $repositoryTaxCategory;
        $this->repositoryTaxon = $repositoryTaxon;
        $this->slugGenerator = $slugGenerator;
        $this->imageUploader = $imageUploader;
        $this->kernelProjectDir = $kernelProjectDir;
        $this->appFtpImagesDir = $appFtpImagesDir;
    }

    public function reporting(UploadedFile $file): array
    {
        try {
            $this->cleanDir();
            $fileName = md5(uniqid()).'.'.$file->getClientOriginalExtension();
            $file->move($this->getDir(), $fileName);

            $taxons = $this->getTaxons();
            $listeReference = [];
            $listeReferenceTHS = [];
            $newReferences = [];
            $updated = [];
            $notUpdated = [];
            $supplierReferenceDuplicate = [];
            $thsSageReferenceDuplicate = [];
            $thsReferenceDuplicate = [];
            $requiredFields = [];
            $line = 1;

            foreach ($this->readFile($fileName) as $data) {
                if ('' == $data[27]) {
                    throw new \Exception("La référence THS '$data[26]' ne contient pas de famille");
                }

                if (!isset($taxons[mb_strtolower($data[27])])) {
                    throw new \Exception("La référence THS '$data[26]' contient une famille qui n'existe pas");
                }

                $listeReferenceTHS[] = $data[26];
                $listeReference[$data[26]] = $data[3];
            }

            if (sizeof($listeReferenceTHS) > 0) {
                /**
                 * We get all existing reference and compare if couple code / thsReferenceSage exist.
                 */
                $products = $this->repositoryProduct->findByCodes($listeReferenceTHS);

                /** @var Product $_product */
                foreach ($products as $_product) {
                    if ($listeReference[$_product->getCode()] != $_product->getThsReferenceSage()) {
                        throw new \Exception("La référence THS '{$_product->getCode()}' ne contient pas la même référence Sage. Référence BDD = {$_product->getThsReferenceSage()}. Référence fichier = {$listeReference[$_product->getCode()]}.");
                    }
                }
            }

            foreach ($this->readFile($fileName) as $data) {
                $data = array_map(function ($value) {
                    return utf8_encode($value);
                }, $data);
                ++$line;
                $supplier = $this->getSupplier($data[1]);

                // Check required fields A, B, C, D, E, F, G, K
                if ('' == $data[0]
                    || '' == $data[1]
                    || '' == $data[2]
                    || '' == $data[3]
                    || '' == $data[4]
                    || '' == $data[5]
                    || '' == $data[6]
                    || '' == $data[10]
                ) {
                    $requiredFields[$line]['data'] = $data;
                    $requiredFields[$line]['fields'] = [];

                    if ('' == $data[0]) {
                        $requiredFields[$line]['fields'][] = 'A';
                    }

                    if ('' == $data[1]) {
                        $requiredFields[$line]['fields'][] = 'B';
                    }

                    if ('' == $data[2]) {
                        $requiredFields[$line]['fields'][] = 'C';
                    }

                    if ('' == $data[3]) {
                        $requiredFields[$line]['fields'][] = 'D';
                    }

                    if ('' == $data[4]) {
                        $requiredFields[$line]['fields'][] = 'E';
                    }

                    if ('' == $data[5]) {
                        $requiredFields[$line]['fields'][] = 'F';
                    }

                    if ('' == $data[6]) {
                        $requiredFields[$line]['fields'][] = 'G';
                    }

                    if ('' == $data[10]) {
                        $requiredFields[$line]['fields'][] = 'K';
                    }

                    continue;
                }

                $data = $this->handleData($supplier, $data);

                /*
                 * Check if there isn't duplicate supplier reference.
                 */
                $supplierReferenceDuplicate[$data[0]]['reference'] = $data[0];
                $supplierReferenceDuplicate[$data[0]]['data'][$line] = $data;

                /*
                 * Check if there isn't duplicate THS reference Sage.
                 */
                $thsSageReferenceDuplicate[$data[3]]['reference'] = $data[3];
                $thsSageReferenceDuplicate[$data[3]]['data'][$line] = $data;

                /*
                 * Check if there isn't duplicate THS reference.
                 */
                $thsReferenceDuplicate[$data[26]]['reference'] = $data[26];
                $thsReferenceDuplicate[$data[26]]['data'][$line] = $data;

                /** @var Product $objet */
                $product = $this->repositoryProduct->findOneBy(['code' => $data[26]]);

                if (!$product) {
                    $newReferences[] = $data;
                    continue;
                }

                $update = [
                    'description' => false,
                    'price' => false,
                ];

                if ($product->getShortDescription() != $data[29] || $product->getDescription() != $data[28]) {
                    $update['description'] = true;
                    $update['old_short_description'] = $product->getShortDescription();
                    $update['old_description'] = $product->getDescription();
                    $update['data'] = $data;
                    $updated[] = $update;
                } else {
                    $notUpdated[] = $data;
                }
            }

            /*
             * Delete references which isn't duplicate
             */
            foreach ($supplierReferenceDuplicate as $key => $reference) {
                if (count($reference['data']) <= 1) {
                    unset($supplierReferenceDuplicate[$key]);
                }
            }

            foreach ($thsSageReferenceDuplicate as $key => $reference) {
                if (count($reference['data']) <= 1) {
                    unset($thsSageReferenceDuplicate[$key]);
                }
            }

            foreach ($thsReferenceDuplicate as $key => $reference) {
                if (count($reference['data']) <= 1) {
                    unset($thsReferenceDuplicate[$key]);
                }
            }
        } catch (\Exception $e) {
            return [
                'error' => $e->getMessage(),
            ];
        }

        return [
            'requiredFields' => $requiredFields,
            'supplierReferenceDuplicate' => $supplierReferenceDuplicate,
            'thsSageReferenceDuplicate' => $thsSageReferenceDuplicate,
            'thsReferenceDuplicate' => $thsReferenceDuplicate,
            'newReferences' => $newReferences,
            'updated' => $updated,
            'notUpdated' => $notUpdated,
            'fileName' => $fileName,
        ];
    }

    /**
     * @throws \Exception
     */
    public function export(string $fileName): string
    {
        $outputFileName = $this->getDir().'/export.csv';
        $fp = fopen($outputFileName, 'w+');

        foreach ($this->readFile($fileName) as $data) {
            $supplier = $this->getSupplier($data[1]);

            // A, B, C, D, E, F, G, K
            if ('' == $data[0]
                || '' == $data[1]
                || '' == $data[2]
                || '' == $data[3]
                || '' == $data[4]
                || '' == $data[5]
                || '' == $data[6]
                || '' == $data[10]
            ) {
                continue;
            }

            $data = $this->handleData($supplier, $data);

            $convert = [7, 8, 9, 12, 14, 15, 16, 17];

            foreach ($convert as $index) {
                $data[$index] = str_replace('.', ',', $data[$index]);
            }

            fputcsv($fp, $data, ';');
        }

        fclose($fp);

        return $outputFileName;
    }

    /**
     * @throws \Exception
     */
    public function import(UploadedFile $file)
    {
        $connection = $this->entityManager->getConnection();

        try {
            $this->cleanDir();
            $fileName = md5(uniqid()).'.'.$file->getClientOriginalExtension();
            $file->move($this->getDir(), $fileName);

            $taxons = $this->getTaxons();
            $listeReference = [];
            $listeReferenceTHS = [];

            foreach ($this->readFile($fileName) as $data) {
                $data = array_map(function ($value) {
                    return utf8_encode($value);
                }, $data);
                $imageExist = false;

                foreach ($this->appFtpImagesDir as $imageDir) {
                    if (
                        file_exists($imageDir.'/'.$data[0].'.jpg')
                        || file_exists($imageDir.'/'.$data[23].'.jpg')
                        || file_exists($imageDir.'/'.$data[26].'.jpg')
                    ) {
                        $imageExist = true;
                        break;
                    }
                }

                if (!$imageExist) {
                    throw new \Exception("Aucune image n'est associé à la référence THS '$data[26]'");
                }

                if ('' == $data[27]) {
                    throw new \Exception("La référence THS '$data[26]' ne contient pas de famille");
                }

                if (!isset($taxons[mb_strtolower($data[27])])) {
                    throw new \Exception("La référence THS '$data[26]' contient une famille qui n'existe pas");
                }

                $listeReferenceTHS[] = $data[26];
                $listeReference[$data[26]] = $data[3];
            }

            if (sizeof($listeReferenceTHS) > 0) {
                /**
                 * We get all existing reference and compare if cuple code / thsReferenceSage exist.
                 */
                $products = $this->repositoryProduct->findByCodes($listeReferenceTHS);

                /** @var Product $_product */
                foreach ($products as $_product) {
                    if ($listeReference[$_product->getCode()] != $_product->getThsReferenceSage()) {
                        throw new \Exception("La référence THS '{$_product->getCode()}' ne contient pas la même référence Sage.");
                    }
                }
            }

            try {
                $connection->beginTransaction();
                $i = 0;

                foreach ($this->readFile($fileName) as $data) {
                    $data = array_map(function ($value) {
                        return utf8_encode($value);
                    }, $data);
                    $supplier = $this->getSupplier($data[1]);

                    // A, B, C, D, E, F, G, K
                    if ('' == $data[0]
                        || '' == $data[1]
                        || '' == $data[2]
                        || '' == $data[3]
                        || '' == $data[4]
                        || '' == $data[5]
                        || '' == $data[6]
                        || '' == $data[10]
                    ) {
                        continue;
                    }

                    $data = $this->handleData($supplier, $data);

                    if ('' == $data[51]) {
                        $data[51] = false;
                    }

                    /** @var Product $product */
                    $product = $this->repositoryProduct->findOneBy(['code' => $data[26]]);
                    $isNewProduct = false;
                    $price = (int) str_replace('.', ',', ((float) str_replace(',', '.', $data[14]) * 100));

                    if (!$product) {
                        $product = $this->repositoryProduct->findOneBy(['thsReferenceSage' => $data[3]]);

                        if (!$product) {
                            /** @var Channel $channel */
                            $channel = $this->repositoryChannel->findOneBy(['code' => 'ECOMMERCE']);

                            $product = new Product();
                            $product->setCurrentLocale('fr_FR');
                            $product->addChannel($channel);
                            $product->setCode($data[26]);
                            $product->setName($data[29]);
                            $product->setMetaDescription(mb_strtolower($data[20]).' - '.$data[28]);
                            $product->setThsReferenceSage($data[3]);
                            $product->setManufacturerReference($data[25]);
                            $product->setThsReferenceUrl($this->slugGenerator->generate($data[24]));
                            $product->setSlug($this->slugGenerator->generate($data[20].' '.$data[24]));
                            $product->setShortDescription($data[29]);
                            $product->setDescription($data[28]);
                            $product->setMinimumOrder('1' == $data[52]);
                            $product->setNoIndex('1' == $data[56]);
                            $product->activate();
                            $product->setTitle($data[21]);

                            /**
                             * Taxons.
                             */
                            $taxon = $taxons[mb_strtolower($data[27])];
                            $product->setMainTaxon($taxon);

                            /** @var ProductTaxon $productTaxon */
                            $productTaxon = $this->factoryProductTaxon->createNew();
                            $productTaxon->setTaxon($taxon);
                            $productTaxon->setProduct($product);
                            $product->addProductTaxon($productTaxon);

                            /** @var ProductTaxon $productTaxon */
                            $productTaxon = $this->factoryProductTaxon->createNew();
                            $productTaxon->setTaxon($this->repositoryTaxon->findOneBy(['code' => 'ARTICLE']));
                            $productTaxon->setProduct($product);
                            $product->addProductTaxon($productTaxon);
                            /**
                             * Attributes.
                             */
                            $product = $this->addAttributesToProduct($product, 'specificity', $data[31]);
                            $product = $this->addAttributesToProduct($product, 'dimension_a', $data[36]);
                            $product = $this->addAttributesToProduct($product, 'dimension_b', $data[37]);
                            $product = $this->addAttributesToProduct($product, 'material', $data[39]);
                            $product = $this->addAttributesToProduct($product, 'maximum_pressure', $data[40]);
                            $product = $this->addAttributesToProduct($product, 'maximum_temperature', $data[41]);
                            $product = $this->addAttributesToProduct($product, 'nominal_diameter', $data[38]);
                            $product = $this->addAttributesToProduct($product, 'metric_reference_input', $data[42]);
                            $product = $this->addAttributesToProduct($product, 'sex_input', $data[43]);
                            $product = $this->addAttributesToProduct($product, 'additional_precision_input', $data[44]);
                            $product = $this->addAttributesToProduct($product, 'metric_reference_output', $data[45]);
                            $product = $this->addAttributesToProduct($product, 'sex_output', $data[46]);
                            $product = $this->addAttributesToProduct($product, 'additional_precision_output', $data[47]);
                            $product = $this->addAttributesToProduct($product, 'caliber_piece', $data[49]);
                            $product = $this->addAttributesToProduct($product, 'maximum_flow', $data[50]);

                            /** @var ProductVariant $productVariant */
                            $productVariant = $this->factoryProductVariant->createNew();

                            $productVariant->setName($data[29]);
                            $productVariant->setCode($data[26]);
                            $productVariant->setPosition(1);
                            $productVariant->setProduct($product);
                            $productVariant->setWeight((float) str_replace(',', '.', $data[35]));
                            $productVariant->setTaxCategory($this->repositoryTaxCategory->findOneBy(['code' => 'SALE20']));
                            $productVariant->setShippingCategory($this->repositoryShippingCategory->findOneBy(['code' => '1' == $data[51] ? 'OUTSIZE' : 'REGULAR']));

                            /** @var ChannelPricing $channelPricing */
                            $channelPricing = $this->factoryChannelPricing->createNew();
                            $channelPricing->setChannelCode($channel->getCode());
                            $channelPricing->setProductVariant($productVariant);
                            $channelPricing->setPrice($price);
                            $channelPricing->setOriginalPrice($price);

                            $productVariant->addChannelPricing($channelPricing);
                            $product->addVariant($productVariant);

                            $product->resetCommercialDocumentations();
                            $product->resetTechnicalBursts();
                            $product->resetTechnicalDocumentations();

                            $this->entityManager->persist($product);
                            $isNewProduct = true;
                        } else {
                            if ($product->getCode() === $product->getThsReferenceSage()) {
                                $product->setManufacturerReference($data[25]);
                                $product->setNoIndex('1' == $data[56]);
                                /** @var ProductVariant $productVariant */
                                $productVariant = $product->getVariants()->first();
                                /** @var ChannelPricing $channelPricing */
                                $channelPricing = $productVariant->getChannelPricings()->first();
                                $channelPricing->setPrice($price);
                                $channelPricing->setOriginalPrice($price);
                                $product->setImported(true);
                                $product->activate();
                            }
                        }
                    } else {
                        if ($product->getCode() === $product->getThsReferenceSage()) {
                            $product->setManufacturerReference($data[25]);
                            $product->setNoIndex('1' == $data[56]);
                            /** @var ProductVariant $productVariant */
                            $productVariant = $product->getVariants()->first();
                            /** @var ChannelPricing $channelPricing */
                            $channelPricing = $productVariant->getChannelPricings()->first();
                            $channelPricing->setPrice($price);
                            $channelPricing->setOriginalPrice($price);
                            $product->setImported(true);
                            $product->activate();
                        }
                    }

                    if ($product && ($isNewProduct || 0 === $product->getImages()->count())) {
                        foreach ($this->appFtpImagesDir as $imageDir) {
                            $imagePath = null;

                            if ('' != $data[0] && file_exists($imageDir.'/'.$data[0].'.jpg')) {
                                $imagePath = $imageDir.'/'.$data[0].'.jpg';
                            } elseif ('' != $data[23] && file_exists($imageDir.'/'.$data[23].'.jpg')) {
                                $imagePath = $imageDir.'/'.$data[23].'.jpg';
                            } elseif ('' != $data[26] && file_exists($imageDir.'/'.$data[26].'.jpg')) {
                                $imagePath = $imageDir.'/'.$data[26].'.jpg';
                            }

                            if (is_null($imagePath)) {
                                continue;
                            }

                            $uploadedImage = new UploadedFile($imagePath, basename($imagePath));

                            /** @var ImageInterface $productImage */
                            $productImage = $this->factoryProductImage->createNew();
                            $productImage->setFile($uploadedImage);
                            $productImage->setType('jpg');
                            $this->imageUploader->upload($productImage);
                            $product->addImage($productImage);
                        }
                    }

                    if (0 === ++$i % 200) {
                        $this->entityManager->flush();
                    }
                }

                $this->entityManager->flush();
                $connection->commit();
            } catch (\Exception $e) {
                $connection->rollBack();

                throw $e;
            }
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * @param string|bool $value
     */
    public function addAttributesToProduct(Product $product, string $code, $value): Product
    {
        /** @var AttributeValueInterface $attributeValue */
        $attributeValue = $this->factoryAttributeValue->createNew();
        $attribute = $this->repositoryProductAttribute->findOneBy(['code' => $code]);
        $attributeValue->setAttribute($attribute);
        $attributeValue->setValue($value);
        $attributeValue->setLocaleCode('fr_FR');
        $product->addAttribute($attributeValue);

        return $product;
    }

    /**
     * @throws \Exception
     */
    protected function getSupplier(string $code): Supplier
    {
        $supplier = $this->repositorySupplier->findOneBy(['code' => $code]);

        if (!$supplier) {
            throw new \Exception("Le code fournisseur '$code' n'existent pas en base de données");
        }

        return $supplier;
    }

    protected function getTaxons(): array
    {
        $listTaxons = $this->repositoryTaxon->findAll();
        $taxons = [];

        /** @var Taxon $taxon */
        foreach ($listTaxons as $taxon) {
            $taxons[mb_strtolower($taxon->getCode())] = $taxon;
        }

        return $taxons;
    }

    /**
     * @throws \Exception
     */
    protected function handleData(Supplier $supplier, array $data): array
    {
        $price = $this->convertToDouble($data[11]);
        $coefficients = $this->repositorySupplierCoefficient->findBySupplierAndPrice($supplier, $price);

        if (count($coefficients) > 1) {
            throw new \Exception('Il y a plusieurs coefficient pour le fournisseur '.$data[1].' et le montant '.$data[11]);
        } elseif (1 == count($coefficients)) {
            $coeff = $coefficients[0]->getCoefficient();
        } else {
            $coeff = 1;
        }

        /**
         * Calculate coefficient sale for THS, THS margin in % and €
         * H = F - ( F * G / 100 )
         * 7 = 5 - ( 5 * 6 / 100 ).
         */
        $f = $this->convertToDouble($data[5]);
        $g = $this->convertToDouble($data[6]);
        $h = $data[7] = round($f - ($f * $g / 100), 2);

        $data[8] = 0 == $h ? 0 : round($price / $h, 2);
        $data[12] = round($price - $h, 2);
        $data[9] = 0 == $price ? 0 : round($data[12] * 100 / $price, 2);

        /**
         * Calculate with THS coefficient sale.
         */
        $webPrice = $price * $coeff;
        $data[14] = round($webPrice, 2);
        $o = round($webPrice, 2);

        /*
         * P = O / H
         * 15 = 14 / 7;
         */
        $data[15] = 0 == $h ? 0 : round($o / $h, 2);

        /*
         * R = O - H
         * 17 = 14 - 7
         */
        $data[17] = round($o - $h, 2);
        $r = round($o - $h, 2);

        /*
         * Q = R x 100 / O
         * 16 = 17 * 100 / 14
         */
        $data[16] = 0 == $o ? 0 : round($r * 100 / $o, 2);

        return $data;
    }

    protected function readFile(string $fileName): iterable
    {
        if (false !== ($handle = fopen($this->getDir().'/'.$fileName, 'r'))) {
            while (false !== ($data = fgetcsv($handle, 0, ';'))) {
                yield $data;
            }

            fclose($handle);
        }
    }

    protected function convertToDouble(string $double): float
    {
        return (float) str_replace(',', '.', $double);
    }

    protected function cleanDir(): void
    {
        $finder = new Finder();
        $finder->name('*.csv');
        $fs = new Filesystem();

        foreach ($finder->in($this->getDir()) as $file) {
            $fs->remove($file);
        }
    }

    protected function getDir(): string
    {
        return $this->kernelProjectDir.'/data/product_reporting';
    }
}
