<?php

declare(strict_types=1);

namespace App\Service;

use App\Entity\Setting;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;

class SettingService
{
    /** @var RepositoryInterface */
    private $repositorySetting;

    /**
     * ConfigurationService constructor.
     *
     * @param RepositoryInterface $repositorySetting
     */
    public function __construct(RepositoryInterface $repositorySetting)
    {
        $this->repositorySetting = $repositorySetting;
    }

    /**
     * @return Setting|null
     */
    public function getMinimumManagementFees(): ?Setting
    {
        return $this->repositorySetting->findOneBy([
            'active' => true,
            'code' => Setting::MINIMUM_MANAGEMENT_FEES,
        ]);
    }

    /**
     * @param OrderInterface $order
     *
     * @return int|null
     */
    public function getManagementFees(OrderInterface $order): ?int
    {
        /** @var Setting $minimumManagementFees */
        $minimumManagementFees = $this->getMinimumManagementFees();

        /** @var Setting $managementFees */
        $managementFees = $this->repositorySetting->findOneBy([
            'active' => true,
            'code' => Setting::MANAGEMENT_FEES,
        ]);

        if (!$minimumManagementFees || !$managementFees) {
            return null;
        }

        $minimum = (int) $minimumManagementFees->getContent();
        $fees = (int) $managementFees->getContent();

        if ($minimum < $order->getItemsTotal()) {
            return null;
        }

        return $fees;
    }

    /**
     * @return int|null
     */
    public function getShippingFees(): ?int
    {
        /** @var Setting $shippingFees */
        $shippingFees = $this->repositorySetting->findOneBy([
            'active' => true,
            'code' => Setting::MINIMUM_SHIPPING_FEES,
        ]);

        if (!$shippingFees) {
            return null;
        }

        return (int) $shippingFees->getContent();
    }

    /**
     * @return string
     */
    public function getHolidayMessage(): ?string
    {
        /** @var Setting $setting */
        $setting = $this->repositorySetting->findOneBy([
            'active' => true,
            'code' => Setting::HOLIDAY,
        ]);

        if (!$setting) {
            return null;
        }

        return $setting->getContent();
    }

    /**
     * @return int|null
     */
    public function getMinimumNPayments(): ?int
    {
        /** @var Setting $setting */
        $setting = $this->repositorySetting->findOneBy([
            'active' => true,
            'code' => Setting::MINIMUM_N_PAYMENTS,
        ]);

        if (!$setting) {
            return null;
        }

        return (int) $setting->getContent();
    }
}
