<?php

declare(strict_types=1);

namespace App\Command;

use App\Importer\ArticleImporter;
use App\Importer\AssociationImporter;
use App\Importer\ImporterInterface;
use App\Importer\OrderImporter;
use App\Importer\OrderLineImporter;
use App\Importer\PressureWasherImporter;
use App\Importer\PumpImporter;
use App\Importer\UserImporter;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

/**
 * Class ImportCommand.
 */
class ImportCommand extends Command
{
    protected static $defaultName = 'app:import';

    /**
     * @var array
     */
    private $types;

    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    /**
     * @var ArticleImporter
     */
    private $articleImporter;

    /**
     * @var ArticleImporter
     */
    private $associationImporter;

    /**
     * @var OrderImporter
     */
    private $orderImporter;

    /**
     * @var OrderLineImporter
     */
    private $orderLineImporter;

    /**
     * @var PressureWasherImporter
     */
    private $pressureWasherImporter;

    /**
     * @var PumpImporter
     */
    private $pumpImporter;

    /**
     * @var UserImporter
     */
    private $userImporter;

    /**
     * ImportPumpCommand constructor.
     *
     * @param EntityManagerInterface $entityManager
     * @param ArticleImporter        $articleImporter
     * @param AssociationImporter    $associationImporter
     * @param OrderImporter          $orderImporter
     * @param OrderLineImporter      $orderLineImporter
     * @param PressureWasherImporter $pressureWasherImporter
     * @param PumpImporter           $pumpImporter
     * @param UserImporter           $userImporter
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        ArticleImporter $articleImporter,
        AssociationImporter $associationImporter,
        OrderImporter $orderImporter,
        OrderLineImporter $orderLineImporter,
        PressureWasherImporter $pressureWasherImporter,
        PumpImporter $pumpImporter,
        UserImporter $userImporter
    ) {
        $this->entityManager = $entityManager;
        $this->articleImporter = $articleImporter;
        $this->associationImporter = $associationImporter;
        $this->orderImporter = $orderImporter;
        $this->orderLineImporter = $orderLineImporter;
        $this->pressureWasherImporter = $pressureWasherImporter;
        $this->pumpImporter = $pumpImporter;
        $this->userImporter = $userImporter;
        $this->types = [
            'article' => [
                'importer' => $this->articleImporter,
                'nb' => 2000,
            ],
            'association' => [
                'importer' => $this->associationImporter,
                'nb' => 2000,
            ],
            'order' => [
                'importer' => $this->orderImporter,
                'nb' => 100,
            ],
            'order_line' => [
                'importer' => $this->orderLineImporter,
                'nb' => 2000,
            ],
            'pressure_washer' => [
                'importer' => $this->pressureWasherImporter,
                'nb' => 2000,
            ],
            'pump' => [
                'importer' => $this->pumpImporter,
                'nb' => 2000,
            ],
            'user' => [
                'importer' => $this->userImporter,
                'nb' => 2000,
            ],
        ];

        parent::__construct();
    }

    protected function configure()
    {
        $this->setDescription('Import article from THS V1')
            ->addArgument('type', InputArgument::REQUIRED, 'Type to import')
            ->addArgument('file', InputArgument::REQUIRED, 'File to import')
        ;
    }

    private function convert($size)
    {
        $unit = array('b', 'kb', 'mb', 'gb', 'tb', 'pb');

        return @round($size / pow(1024, ($i = floor(log($size, 1024)))), 2).' '.$unit[$i];
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int|null|void
     *
     * @throws \Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $io = new SymfonyStyle($input, $output);
        $type = $input->getArgument('type');
        $file = $input->getArgument('file');
        $io->note(sprintf('You passed an argument: %s', $file));
        $i = 1;
        $this->entityManager->beginTransaction();

        foreach ($this->read($file) as $data) {
            $this->getImporter($type)->import($data);
            // $io->writeln(sprintf('Import %s %s', $data[4], $data[0]));

            if (0 === $i++ % $this->types[$type]['nb']) {
                $this->entityManager->flush();
                $this->entityManager->clear();
                $io->writeln($this->convert(memory_get_usage())."-$i");
            }
        }

        $this->entityManager->flush();
        $this->entityManager->commit();
        $io->success('Import finished.');
    }

    /**
     * @param $file
     *
     * @return \Generator
     */
    private function read($file): \Generator
    {
        if (false !== ($handle = fopen($file, 'r'))) {
            while (false !== ($data = fgetcsv($handle, 0, ';'))) {
                yield $data;
            }

            fclose($handle);
        }
    }

    /**
     * @param string $type
     *
     * @return ImporterInterface
     *
     * @throws \Exception
     */
    private function getImporter(string $type): ImporterInterface
    {
        if (!array_key_exists($type, $this->types)) {
            throw new \Exception(sprintf('%s type unknown.', $type));
        }

        return $this->types[$type]['importer'];
    }
}
