<?php

declare(strict_types=1);

namespace App\Command;

use App\Entity\Product;
use Doctrine\ORM\EntityManagerInterface;
use Sylius\Component\Product\Generator\SlugGeneratorInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class ProductSlugifyCommand extends Command
{
    protected static $defaultName = 'app:product:slugify';

    /** @var EntityManagerInterface */
    private $entityManager;

    /** @var RepositoryInterface */
    private $productRepository;

    /** @var SlugGeneratorInterface */
    private $slugGenerator;

    /**
     * ProductSlugifyCommand constructor.
     *
     * @param EntityManagerInterface $entityManager
     * @param RepositoryInterface    $productRepository
     * @param SlugGeneratorInterface $slugGenerator
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        RepositoryInterface $productRepository,
        SlugGeneratorInterface $slugGenerator
    ) {
        $this->entityManager = $entityManager;
        $this->productRepository = $productRepository;
        $this->slugGenerator = $slugGenerator;

        parent::__construct();
    }

    protected function configure()
    {
        $this->setDescription('Slugify products.');
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int|void|null
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $io = new SymfonyStyle($input, $output);
        $io->text('Start slugify');
        $result = $this->productRepository->findAllIterate();
        $i = 0;

        /** @var Product $product */
        foreach ($result as $product) {
            $mainTaxon = $product->getMainTaxon();

            switch ($mainTaxon->getCode()) {
                case 'NHP':
                case 'POMPE':
                    $slug = $mainTaxon->getName().' '.$product->getName();
                    break;
                default:
                    $slug = $mainTaxon->getName().' '.$product->getThsReferenceUrl();
            }

            $product->setSlug($this->slugGenerator->generate(mb_strtolower($slug)));

            if (0 === ++$i % 500) {
                $io->text('Count '.$i);
                $this->entityManager->flush();
            }
        }

        $io->text('End with '.$i);
        $this->entityManager->flush();
    }
}
