<?php

declare(strict_types=1);

namespace App\Controller;

use App\Entity\Customer;
use App\Entity\Order;
use App\Form\Type\AdminOrderQuotationType;
use App\Form\Type\AdminOrderTicketType;
use App\Model\AdminOrderQuotation;
use App\Model\AdminOrderTicket;
use App\Service\IdentifierService;
use App\Service\OrderService;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use Psr\Log\LoggerInterface;
use Sylius\AdminOrderCreationPlugin\Factory\OrderFactoryInterface;
use Sylius\AdminOrderCreationPlugin\Form\Type\NewOrderType;
use Sylius\Component\Core\Model\AddressInterface;
use Sylius\Component\Core\OrderPaymentStates;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Finder\Finder;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\HttpKernel\KernelInterface;

class AdminOrderController extends AbstractController
{
    /** @var OrderFactoryInterface */
    private $orderFactory;
    private $identifierService;

    /**
     * AdminOrderController constructor.
     */
    public function __construct(OrderFactoryInterface $orderFactory, IdentifierService $identifierService)
    {
        $this->orderFactory = $orderFactory;
        $this->identifierService = $identifierService;
    }

    public function cart(Request $request, OrderService $orderService): Response
    {
        $customerId = $request->attributes->get('customerId');
        $channelCode = $request->attributes->get('channelCode');

        $order = $this->orderFactory->createForCustomerAndChannel($customerId, $channelCode);

        $form = $this->createForm(NewOrderType::class, $order);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            try {
                $order = $form->getData();
                $order = $orderService->createCart($order, $this->getUser());

                return $this->redirectToRoute('sylius_admin_order_show', ['id' => $order->getId()]);
            } catch (\Exception $e) {
                $this->addFlash('error', $e->getMessage());
            }
        }

        return $this->render('SyliusAdminOrderCreationPlugin:Order:create.html.twig', [
            'order' => $order,
            'form' => $form->createView(),
        ]);
    }

    public function ticket(Request $request, OrderService $orderService): Response
    {
        $form = $this->createForm(AdminOrderTicketType::class, new AdminOrderTicket());
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var AdminOrderTicket $adminOrderTicket */
            $adminOrderTicket = $form->getData();

            try {
                $order = $orderService->createTicket($adminOrderTicket, $this->getUser());

                return $this->redirectToRoute('sylius_admin_order_show', ['id' => $order->getId()]);
            } catch (\Exception $e) {
                $this->addFlash('error', $e->getMessage());
            }
        }

        return $this->render('SyliusAdminBundle:Order:ticket.html.twig', [
            'form' => $form->createView(),
        ]);
    }

    public function quotation(Request $request, OrderService $orderService): Response
    {
        $form = $this->createForm(AdminOrderQuotationType::class, new AdminOrderQuotation());
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var AdminOrderQuotation $data */
            $data = $form->getData();

            try {
                $order = $orderService->createQuotation($data, $this->getUser());

                return $this->redirectToRoute('sylius_admin_order_show', ['id' => $order->getId()]);
            } catch (\Exception $e) {
                $this->addFlash('error', $e->getMessage());
            }
        }

        return $this->render('SyliusAdminBundle:Order:quotation.html.twig', [
            'form' => $form->createView(),
        ]);
    }

    public function viewQuotation(Order $order, OrderService $orderService): Response
    {
        return $this->file($orderService->getQuotationDir().'/'.$order->getQuotationPdf(), null, ResponseHeaderBag::DISPOSITION_INLINE);
    }

    public function downloadExport(KernelInterface $kernel)
    {
        $dir = $kernel->getVarDir().'/export_sage';
        $finder = new Finder();
        $finder->files()->in($dir)->sortByModifiedTime()->reverseSorting();

        if (!$finder->hasResults()) {
            $this->addFlash('warning', 'Aucun export à télécharger');

            return $this->redirectToRoute('sylius_admin_order_index');
        }

        foreach ($finder as $file) {
            return $this->file($file->getRealPath());
        }
    }

    public function exportSage(
        Request $request,
        EntityManagerInterface $entityManager,
        LoggerInterface $logger,
        KernelInterface $kernel
    ): RedirectResponse {
        try {
            $entityManager->beginTransaction();
            $data = $request->request->get('ids', []);
            $dir = $kernel->getVarDir();
            $fp = fopen($dir.'/export_sage/'.date('ymdhis').'.txt', 'w');
            $nb = count($data);
            $i = 0;

            if (0 === $nb) {
                throw new Exception('Aucun client à exporter.');
            }

            foreach ($data as $id) {
                /** @var Order $order */
                $order = $entityManager->getRepository(Order::class)->findOneBy(['id' => (int) $id]);

                if (!$order) {
                    continue;
                }

                /** @var Customer $customer */
                $customer = $order->getCustomer();

                if (!$customer->getIdentifier()) {
                    $customer->setIdentifier($this->identifierService->createIdentifier($customer));
                }

                /* @var AddressInterface $address */
                $addresses = $customer->getAddresses();
                $address = $addresses->last();

                $content = ($customer->getIdentifier() ?? '')."\t";
                $name = [];

                if ('' !== $address->getCompany()) {
                    $name[] = $this->transform($address->getCompany());
                }

                if ('' !== $address->getLastName()) {
                    $name[] = $this->transform($address->getLastName());
                }

                if ('' !== $address->getFirstName()) {
                    $name[] = $this->transform($address->getFirstName());
                }

                $phones = array_filter([
                    $this->cleanPhoneNumber($address->getPhoneNumber()),
                    $this->cleanPhoneNumber($address->getMobilePhoneNumber()),
                ]);
                $phone = implode('/', $phones);

                $content .= strtoupper(implode(' ', $name))."\t";
                $content .= "41100000\t"
                    .($address->getCivility() ? $this->transform($address->getCivility()) : '')."\t"
                    .($customer->getIdentifier() ?? '')."\t"
                    .($address->getStreet() ? $this->transform($address->getStreet()) : '')."\t\t"
                    .($address->getPostcode() ? $this->transform($address->getPostcode()) : '')."\t"
                    .($address->getCity() ? $this->transform($address->getCity()) : '')."\t"
                    .$this->transform($address->getCountryCode())."\t"
                    .$phone."\t"
                    .$address->getFaxNumber()."\t"
                    .$customer->getEmail()."\t";

                if ($order && OrderPaymentStates::STATE_PAID === $order->getPaymentState()) {
                    /** @var AddressInterface $billingAddress */
                    $billingAddress = $order->getBillingAddress();
                    $billingPhones = array_filter([
                        $this->cleanPhoneNumber($billingAddress->getPhoneNumber()),
                        $this->cleanPhoneNumber($billingAddress->getMobilePhoneNumber()),
                    ]);
                    $billingPhone = implode('/', $billingPhones);

                    $content .= $this->transform($billingAddress->getCompany().' '.$billingAddress->getLastName().' '.$billingAddress->getFirstName())."\t"
                        .($billingAddress->getStreet() ? $this->transform($billingAddress->getStreet()) : '')."\t\t"
                        .($billingAddress->getPostcode() ? $this->transform($billingAddress->getPostcode()) : '')."\t"
                        .($billingAddress->getCity() ? $this->transform($billingAddress->getCity()) : '')."\t"
                        .$this->transform($billingAddress->getCountryCode())."\t"
                        .$billingPhone."\t"
                        .$billingAddress->getFaxNumber()."\t".PHP_EOL;
                } else {
                    $content .= "\t\t\t\t\t\t\t".PHP_EOL;
                }

                fwrite($fp, $content);

                if (++$i < $nb) {
                    fputcsv($fp, []);
                }

                $customer->setExportSage(true);
            }

            fputcsv($fp, []);
            fclose($fp);

            $entityManager->flush();
            $entityManager->commit();
            $this->addFlash('success', 'L\'export Sage a bien été réalisé.');
        } catch (Exception $e) {
            $logger->error($e->getMessage());
            $this->addFlash('error', "Une erreur est survenue: {$e->getMessage()}");
            $entityManager->rollback();
        }

        return $this->redirectToRoute('sylius_admin_order_index');
    }

    /**
     * @throws Exception
     */
    public function exportColis(
        Request $request,
        EntityManagerInterface $entityManager,
        LoggerInterface $logger,
        KernelInterface $kernel
    ): RedirectResponse {
        try {
            $data = $request->request->get('ids', []);
            $dir = $kernel->getVarDir();
            $fp = fopen($dir.'/export_sage/EC_'.date('ymdhis').'.txt', 'wb');
            $nb = count($data);
            $i = 0;

            if (0 === $nb) {
                throw new Exception('Aucun client à exporter.');
            }

            foreach ($data as $id) {
                /** @var Order $order */
                $order = $entityManager->getRepository(Order::class)->findOneBy(['id' => (int) $id]);

                if (!$order) {
                    continue;
                }

                /** @var Customer $customer */
                $customer = $order->getCustomer();

                if (!$customer->getIdentifier()) {
                    $customer->setIdentifier($this->identifierService->createIdentifier($customer));
                }

                $content = ($customer->getIdentifier() ?? '')."\t";

                if ($order) {
                    /** @var AddressInterface $shippingAddress */
                    $shippingAddress = $order->getShippingAddress();
                    $wordWrapAddress = wordwrap($shippingAddress->getStreet(), 32, "\n", true);
                    $addresses = explode("\n", $wordWrapAddress);
                    $addressOne = $addresses[0];
                    $addressTwo = $addresses[1] ?? '';
                    $content .= ($shippingAddress->getCompany() ? $this->transform($shippingAddress->getCompany()) : $this->transform($shippingAddress->getLastName()).', '.$this->transform($shippingAddress->getFirstName()))."\t"
                        .($shippingAddress->getLastName() ? $this->transform($shippingAddress->getLastName()) : '')."\t"
                        .($shippingAddress->getFirstName() ? $this->transform($shippingAddress->getFirstName()) : '')."\t"
                        .($addressOne ? $this->transform($addressOne) : '')."\t"
                        .($addressTwo ? $this->transform($addressTwo) : '')."\t"
                        .($shippingAddress->getPostcode() ? $this->transform($shippingAddress->getPostcode()) : '')."\t"
                        .($shippingAddress->getCity() ? $this->transform($shippingAddress->getCity()) : '')."\t"
                        .($shippingAddress->getPhoneNumber() ?: '')."\t"
                        .($shippingAddress->getMobilePhoneNumber() ?: '')."\t"
                        .($shippingAddress->getFaxNumber() ?: '')."\t"
                        .($customer->getEmail() ? $this->transform($customer->getEmail()) : '')."\t"
                        .($shippingAddress->getCountryCode() ?: '')."\t"
                        .($shippingAddress->getProvinceCode() ?: '')."\t"
                        .($order->getNumber() ?: '')."\t".PHP_EOL;
                }

                fwrite($fp, $content);

                if (++$i < $nb) {
                    fputcsv($fp, []);
                }
            }

            fputcsv($fp, []);
            fclose($fp);

            $this->addFlash('success', 'L\'export Colis a bien été réalisé.');
        } catch (Exception $e) {
            $logger->error($e->getMessage());
            $this->addFlash('error', "Une erreur est survenue: {$e->getMessage()}");
        }

        return $this->redirectToRoute('sylius_admin_order_index');
    }

    /**
     * @throws Exception
     */
    public function exportCommande(
        Request $request,
        EntityManagerInterface $entityManager,
        LoggerInterface $logger,
        KernelInterface $kernel
    ): RedirectResponse {
        try {
            $data = $request->request->get('ids', []);
            $dir = $kernel->getVarDir();
            $fp = fopen($dir.'/export_sage/CDE_'.date('ymdhis').'.txt', 'wb');
            $nb = count($data);

            if (0 === $nb) {
                throw new Exception('Aucune commande à exporter.');
            }

            foreach ($data as $id) {
                /** @var Order $order */
                $order = $entityManager->getRepository(Order::class)->findOneBy(['id' => (int) $id]);

                if (!$order) {
                    continue;
                }

                /** @var Customer $customer */
                $customer = $order->getCustomer();

                if (!$customer->getIdentifier()) {
                    $customer->setIdentifier($this->identifierService->createIdentifier($customer));
                }

                if ($order && OrderPaymentStates::STATE_PAID === $order->getPaymentState()) {
                    $content = [];
                    $orderNumber = $order->getNumber() ? 'VD'.(int) $order->getNumber() : '';

                    if ($order->getPaidPayment() && $order->getPaidPayment()->getUpdatedAt()) {
                        $paidPaymentDate = $order->getPaidPayment()->getUpdatedAt()->format('dmy');
                    }

                    $content[] = ['1', ($customer->getIdentifier() ?? ''), $orderNumber, $orderNumber];
                    $j = 0;
                    $items = $order->getItems();

                    foreach ($items as $item) {
                        if ($item->getVariant() && $item->getVariant()->getProduct() && $item->getVariant()->getProduct()->getThsReferenceSage()) {
                            $thsReferenceSage = $this->transform($item->getVariant()->getProduct()->getThsReferenceSage());
                        }

                        if (0 !== $j) {
                            $content[] = ['', '', $orderNumber, '', ($paidPaymentDate ?? ''), ($thsReferenceSage ?? ''), ($item->getProductName() ? $this->transform($item->getProductName()) : ''), ($item->getUnitPrice() ? $this->formatPrice($item->getUnitPrice()) : 0), ($item->getQuantity() ?: ''), ($order->getSalesman() ? $order->getSalesman()->getFirstName().' '.$order->getSalesman()->getLastName() : 'VENTE-DIRECTE-HP')];
                        } else {
                            $content[0][] = ($paidPaymentDate ?? '');
                            $content[0][] = ($thsReferenceSage ?? '');
                            $content[0][] = ($item->getProductName() ? $this->transform($item->getProductName()) : '');
                            $content[0][] = ($item->getUnitPrice() ? $this->formatPrice($item->getUnitPrice()) : 0);
                            $content[0][] = ($item->getQuantity() ?: '');
                            $content[0][] = ($order->getSalesman() ? $order->getSalesman()->getFirstName().' '.$order->getSalesman()->getLastName() : 'VENTE-DIRECTE-HP');
                        }
                        ++$j;
                    }

                    if ($order->getShippingTotalWithoutTax()) {
                        $content[] = ['', '', $orderNumber, '', ($paidPaymentDate ?? ''), 'PORT1', 'FRAIS DE PORT', $this->formatPrice($order->getShippingTotalWithoutTax()), '1', 'PRESTATION-ADMINISTRATIVE'];
                    }

                    if ($order->getManagementFeesTotal()) {
                        $content[] = ['', '', $orderNumber, '', ($paidPaymentDate ?? ''), 'FRAISGESTION', 'FRAIS DE GESTION DES COMMANDES < 50'.iconv(mb_detect_encoding('€'), 'CP1252', '€'), $this->formatPrice($order->getManagementFees()), '1', 'PRESTATION-ADMINISTRATIVE'];
                    }
                    $content[] = ['', '', $orderNumber, '', ($paidPaymentDate ?? ''), '', 'PAYE PAR : '.($order->getPaidPaymentMethod() ? $this->transform($order->getPaidPaymentMethod()->getName()) : '').' - '.($paidPaymentDate ?? ''), '', '', ''];

                    foreach ($content as $line) {
                        fwrite($fp, implode("\t", $line)."\r\n");
                    }
                }
            }

            fclose($fp);

            $this->addFlash('success', 'L\'export Commande a bien été réalisé.');
        } catch (Exception $e) {
            $logger->error($e->getMessage());
            $this->addFlash('error', "Une erreur est survenue: {$e->getMessage()}");
        }

        return $this->redirectToRoute('sylius_admin_order_index');
    }

    /**
     * @param string $text
     *
     * @return string
     */
    private function transform(?string $text): ?string
    {
        if (!$text) {
            return $text;
        }

        $text = $this->stripAccents($text);
        $text = strtoupper($text);
        $text = trim($text);

        return $text;
    }

    private function stripAccents(string $str): string
    {
        return strtr(utf8_decode($str), utf8_decode('àáâãäçèéêëìíîïñòóôõöùúûüýÿÀÁÂÃÄÇÈÉÊËÌÍÎÏÑÒÓÔÕÖÙÚÛÜÝ'), 'aaaaaceeeeiiiinooooouuuuyyAAAAACEEEEIIIINOOOOOUUUUY');
    }

    private function formatPrice(int $price): string
    {
        return str_replace('.', ',', round($price / 100, 2));
    }

    private function cleanPhoneNumber(?string $phone)
    {
        return $phone ? str_replace(['.', ' '], '', $phone) : null;
    }
}
