<?php

declare(strict_types=1);

namespace App\Controller;

use App\Entity\Order;
use App\Entity\Product;
use App\Service\OrderService;
use Doctrine\DBAL\ConnectionException;
use Doctrine\ORM\EntityManagerInterface;
use Sylius\Component\Order\Context\CompositeCartContext;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

class OrderController extends AbstractController
{
    /** @var CompositeCartContext */
    private $contextCart;

    /** @var RepositoryInterface */
    private $repositoryProduct;

    /** @var RepositoryInterface */
    private $repositoryOrder;

    /** @var EntityManagerInterface */
    private $entityManager;

    /**
     * OrderController constructor.
     */
    public function __construct(
        CompositeCartContext $contextCart,
        RepositoryInterface $repositoryOrder,
        RepositoryInterface $repositoryProduct,
        EntityManagerInterface $entityManager
    ) {
        $this->contextCart = $contextCart;
        $this->repositoryOrder = $repositoryOrder;
        $this->repositoryProduct = $repositoryProduct;
        $this->entityManager = $entityManager;
    }

    public function catalog(): Response
    {
        return $this->render('shop/order/catalog.html.twig');
    }

    public function order(Request $request, string $tokenValue): Response
    {
        /** @var Order $order */
        $order = $this->repositoryOrder->findActiveByOrderToken($tokenValue);

        if ($order) {
            return $this->render('shop/order/already_order.html.twig');
        }

        /** @var Order $order */
        $order = $this->repositoryOrder->findOfferedByOrderToken($tokenValue);

        if (!$order) {
            throw $this->createNotFoundException();
        }

        if ($request->get('from')) {
            $order->setEmailLinkClicked(true);
            $this->entityManager->flush();
        }

        switch ($order->getType()) {
            case Order::TYPE_DEVIS:
                $view = 'shop/order/quotation.html.twig';
                break;
            case Order::TYPE_TICKET:
                $view = 'shop/order/ticket.html.twig';
                break;
            default:
                return $this->redirectToRoute('app_cart_confirm', ['tokenValue' => $tokenValue]);
        }

        return $this->render($view, [
            'cart' => $order,
        ]);
    }

    /**
     * @throws ConnectionException
     */
    public function confirmOrder(string $tokenValue, OrderService $orderService): Response
    {
        /** @var Order $order */
        $order = $this->repositoryOrder->findOfferedByOrderToken($tokenValue);

        if (!$order) {
            throw $this->createNotFoundException();
        }

        $cart = $this->contextCart->getCart();
        $orderService->removeOrder($cart);
        /** @var Order $cart */
        $cart = $this->contextCart->getCart();
        $orderService->createCartFromOrder($cart, $order);

        if (Order::TYPE_PANIER === $order->getType()) {
            $route = 'sylius_shop_cart_summary';
        } else {
            $route = 'sylius_shop_checkout_start';
        }

        return $this->redirectToRoute($route);
    }

    public function multipleAdd(Request $request, OrderService $orderService): Response
    {
        $references = $request->request->get('reference');
        $quantities = $request->request->get('quantity');
        $cart = $this->contextCart->getCart();

        foreach ($references as $key => $reference) {
            /** @var Product $product */
            $product = $this->repositoryProduct->findOneBy(['manufacturerReference' => $reference, 'enabled' => true, 'onSale' => true]);

            if (!$product) {
                break;
            }

            try {
                $cart = $orderService->addProductToOrder($cart, $product, (int) $quantities[$key]);
            } catch (\Exception $e) {
                break;
            }
        }

        $orderService->processOrder($cart);

        return $this->redirectToRoute('sylius_shop_cart_summary');
    }

    public function viewQuotation(Order $order, OrderService $orderService): Response
    {
        return $this->file($orderService->getQuotationDir().'/'.$order->getQuotationPdf(), null, ResponseHeaderBag::DISPOSITION_INLINE);
    }
}
