<?php

declare(strict_types=1);

namespace App\Controller;

use App\Entity\Taxon;
use FOS\RestBundle\View\View;
use Sylius\Bundle\ResourceBundle\Controller\ResourceController;
use Sylius\Component\Product\Model\ProductAttribute;
use Sylius\Component\Resource\ResourceActions;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class ProductController extends ResourceController
{
    public function showAction(Request $request): Response
    {
        $configuration = $this->requestConfigurationFactory->create($this->metadata, $request);

        $this->isGrantedOr403($configuration, ResourceActions::SHOW);
        $resource = $this->findOr404($configuration);

        $this->eventDispatcher->dispatch(ResourceActions::SHOW, $configuration, $resource);

        $view = View::create($resource);

        if ($configuration->isHtmlRequest()) {
            $view
                ->setTemplate($configuration->getTemplate(ResourceActions::SHOW.'.html'))
                ->setTemplateVar($this->metadata->getName())
                ->setData([
                    'configuration' => $configuration,
                    'metadata' => $this->metadata,
                    'compatiblesavec' => $resource->getProductAssociation('ACCESSORIES'), //->getAssociatedProducts(),
                    'compatibles' => $resource->getProductAssociation('NHP'),
                    'resource' => $resource,
                    $this->metadata->getName() => $resource,
                ])
            ;
        }

        return $this->viewHandler->handle($configuration, $view);
    }

    public function indexAction(Request $request): Response
    {
        $configuration = $this->requestConfigurationFactory->create($this->metadata, $request);

        $this->isGrantedOr403($configuration, ResourceActions::INDEX);

        $slug = $request->attributes->get('slug');

        if (!$slug) {
            return parent::indexAction($request);
        }

        /** @var Taxon $taxon */
        $taxon = $this->manager->getRepository(Taxon::class)->findOneBySlug($slug, 'fr_FR');

        if (!$taxon) {
            throw $this->createNotFoundException();
        }

        $taxonToDisplay = [];
        if ($taxon->isDisplaySearchForm()) {
            $taxonToDisplay = $this->manager->getRepository(Taxon::class)
                ->findAllSearchableWithSearchableProduct($this->get('sylius.context.locale')->getLocaleCode());
        }

        if ($taxon->getRedirectionTaxon()) {
            return $this->redirectToRoute('sylius_shop_product_index', [
                'slug' => $taxon->getRedirectionTaxon()->getSlug(),
            ], Response::HTTP_MOVED_PERMANENTLY);
        }

        if ($page = $taxon->getPage()) {
            $resources = ['page' => $page];
            $view = View::create();
            $view->setTemplate('@SyliusShop/Product/indexPage.html.twig')
                ->setTemplateVar($this->metadata->getPluralName())
                ->setData([
                    'configuration' => $configuration,
                    'metadata' => $this->metadata,
                    'resources' => $resources,
                    'taxonToDisplay' => $taxonToDisplay,
                    'displaySearchForm' => $taxon->isDisplaySearchForm(),
                    $this->metadata->getPluralName() => $resources,
                ])
            ;

            return $this->viewHandler->handle($configuration, $view);
        }

        /** @var Taxon $parentTaxon */
        $parentTaxon = $taxon->getParent();

        if ($parentTaxon) {
            $attributes = $this->manager->getRepository(ProductAttribute::class)->findAllBoolean();
            $attributesWithCodeInKey = [];
            $nbNhp = 0;
            $nhps = [];

            /** @var ProductAttribute $attribute */
            foreach ($attributes as $attribute) {
                $products = $this->repository->findByBooleanAttribute($taxon, $attribute->getCode());
                $nbProducts = count($products);

                if (0 < $nbProducts) {
                    $nhps[$attribute->getCode()] = $products;
                    $nbNhp += $nbProducts;
                    $attributesWithCodeInKey[$attribute->getCode()] = $attribute;
                }
            }

            $resources = [
                'nhps' => $nhps,
                'attributes' => $attributesWithCodeInKey,
                'nbNhp' => $nbNhp,
                'all' => 0 < $nbNhp ? [] : $this->repository->findByNothing($taxon),
                'taxon' => $taxon,
            ];

            $view = View::create($resources);
            $view->setTemplate('@SyliusShop/Product/indexNhp.html.twig')
                ->setTemplateVar($this->metadata->getPluralName())
                ->setData([
                    'configuration' => $configuration,
                    'metadata' => $this->metadata,
                    'resources' => $resources,
                    'taxonToDisplay' => $taxonToDisplay,
                    'displaySearchForm' => $taxon->isDisplaySearchForm(),
                    $this->metadata->getPluralName() => $resources,
                ])
            ;

            return $this->viewHandler->handle($configuration, $view);
        }

        return parent::indexAction($request);
    }
}
