<?php

declare(strict_types=1);

namespace App\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Sylius\Component\Core\Model\ImageInterface;
use Sylius\Component\Core\Model\ImagesAwareInterface;
use Sylius\Component\Core\Model\Product as BaseProduct;
use Sylius\Component\Product\Model\ProductAssociation;
use Sylius\Component\Product\Model\ProductTranslationInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Table(name="sylius_product")
 * @ORM\Entity
 */
class Product extends BaseProduct implements ImagesAwareInterface
{
    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @Assert\NotBlank
     *
     * @var string
     */
    private $manufacturerReference;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @Assert\NotBlank
     *
     * @var string
     */
    private $thsReferenceSage;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @Assert\NotBlank
     *
     * @var string
     */
    private $thsReferenceUrl;

    /**
     * @ORM\ManyToOne(targetEntity="Supplier")
     * @ORM\JoinColumn(name="supplier_id", referencedColumnName="id")
     *
     * @Assert\NotBlank
     *
     * @var Supplier
     */
    private $supplier;

    /**
     * @ORM\Column(type="boolean", nullable=false)
     *
     * @var bool
     */
    private $minimumOrder;

    /**
     * @ORM\Column(type="boolean", nullable=false)
     *
     * @var bool
     */
    private $freePort;

    /**
     * @ORM\Column(type="boolean", nullable=false)
     *
     * @var bool
     */
    private $noIndex;

    /**
     * @ORM\Column(type="boolean", nullable=false)
     *
     * @var bool
     */
    private $onSale;

    /**
     * @ORM\Column(type="boolean", nullable=false)
     *
     * @var bool
     */
    private $displayMenu;

    /**
     * @ORM\Column(type="boolean", nullable=false)
     *
     * @var bool
     */
    private $displayPrice;

    /**
     * @ORM\Column(type="boolean", nullable=false)
     *
     * @var bool
     */
    private $noLongerManufactured;

    /**
     * @ORM\Column(type="boolean", nullable=false)
     *
     * @var bool
     */
    private $promotion;

    /**
     * @ORM\Column(type="boolean", nullable=false)
     *
     * @var bool
     */
    private $saleSpareParts;

    /**
     * @ORM\OneToMany(targetEntity="ProductCommercialDocumentation", mappedBy="owner", orphanRemoval=true, cascade={"ALL"})
     *
     * @Assert\Valid
     *
     * @var Collection|ImageInterface[]
     */
    protected $commercialDocumentations;

    /**
     * @ORM\OneToMany(targetEntity="ProductTechnicalDocumentation", mappedBy="owner", orphanRemoval=true, cascade={"ALL"})
     *
     * @Assert\Valid
     *
     * @var Collection|ImageInterface[]
     */
    protected $technicalDocumentations;

    /**
     * @ORM\OneToMany(targetEntity="ProductTechnicalBurst", mappedBy="owner", orphanRemoval=true, cascade={"ALL"})
     *
     * @Assert\Valid
     *
     * @var Collection|ImageInterface[]
     */
    protected $technicalBursts;

    /**
     * @ORM\Column(type="boolean", nullable=false)
     *
     * @var bool
     */
    protected $imported;

    /**
     * @ORM\Column(name="display_search", type="boolean", nullable=false)
     *
     * @var bool
     */
    private $displaySearch;

    /**
     * @var string
     */
    private $url;

    public function __construct()
    {
        parent::__construct();

        $this->minimumOrder = false;
        $this->freePort = false;
        $this->noIndex = false;
        $this->onSale = false;
        $this->displayMenu = false;
        $this->displayPrice = false;
        $this->noLongerManufactured = false;
        $this->promotion = false;
        $this->saleSpareParts = false;
        $this->imported = true;
        $this->displaySearch = false;
        $this->commercialDocumentations = new ArrayCollection();
        $this->addCommercialDocumentation(new ProductCommercialDocumentation());
        $this->technicalDocumentations = new ArrayCollection();
        $this->addTechnicalDocumentation(new ProductTechnicalDocumentation());
        $this->technicalBursts = new ArrayCollection();
        $this->addTechnicalBurst(new ProductTechnicalBurst());
    }

    public function getSupplier(): ?Supplier
    {
        return $this->supplier;
    }

    /**
     * @return Product
     */
    public function setSupplier(Supplier $supplier): self
    {
        $this->supplier = $supplier;

        return $this;
    }

    public function isMinimumOrder(): bool
    {
        return $this->minimumOrder;
    }

    /**
     * @return Product
     */
    public function setMinimumOrder(bool $minimumOrder): self
    {
        $this->minimumOrder = $minimumOrder;

        return $this;
    }

    public function getManufacturerReference(): ?string
    {
        return $this->manufacturerReference;
    }

    /**
     * @return Product
     */
    public function setManufacturerReference(string $manufacturerReference): self
    {
        $this->manufacturerReference = $manufacturerReference;

        return $this;
    }

    public function getThsReferenceSage(): ?string
    {
        return $this->thsReferenceSage;
    }

    /**
     * @return Product
     */
    public function setThsReferenceSage(string $thsReferenceSage): self
    {
        $this->thsReferenceSage = $thsReferenceSage;

        return $this;
    }

    public function getThsReferenceUrl(): ?string
    {
        return $this->thsReferenceUrl;
    }

    /**
     * @return Product
     */
    public function setThsReferenceUrl(string $thsReferenceUrl): self
    {
        $this->thsReferenceUrl = $thsReferenceUrl;

        return $this;
    }

    public function isFreePort(): bool
    {
        return $this->freePort;
    }

    /**
     * @return Product
     */
    public function setFreePort(bool $freePort): self
    {
        $this->freePort = $freePort;

        return $this;
    }

    public function isNoIndex(): bool
    {
        return $this->noIndex;
    }

    /**
     * @return Product
     */
    public function setNoIndex(bool $noIndex): self
    {
        $this->noIndex = $noIndex;

        return $this;
    }

    public function isDisplayMenu(): bool
    {
        return $this->displayMenu;
    }

    /**
     * @return Product
     */
    public function setDisplayMenu(bool $displayMenu): self
    {
        $this->displayMenu = $displayMenu;

        return $this;
    }

    public function isDisplayPrice(): bool
    {
        return $this->displayPrice;
    }

    /**
     * @return Product
     */
    public function setDisplayPrice(bool $displayPrice): self
    {
        $this->displayPrice = $displayPrice;

        return $this;
    }

    public function isOnSale(): bool
    {
        return $this->onSale;
    }

    /**
     * @return Product
     */
    public function setOnSale(bool $onSale): self
    {
        $this->onSale = $onSale;

        return $this;
    }

    public function isNoLongerManufactured(): bool
    {
        return $this->noLongerManufactured;
    }

    /**
     * @return Product
     */
    public function setNoLongerManufactured(bool $noLongerManufactured): self
    {
        $this->noLongerManufactured = $noLongerManufactured;

        return $this;
    }

    public function isPromotion(): bool
    {
        return $this->promotion;
    }

    /**
     * @return Product
     */
    public function setPromotion(bool $promotion): self
    {
        $this->promotion = $promotion;

        return $this;
    }

    public function isSaleSpareParts(): bool
    {
        return $this->saleSpareParts;
    }

    /**
     * @return Product
     */
    public function setSaleSpareParts(bool $saleSpareParts): self
    {
        $this->saleSpareParts = $saleSpareParts;

        return $this;
    }

    public function getCommercialDocumentations(): Collection
    {
        if (0 === $this->commercialDocumentations->count()) {
            $this->addCommercialDocumentation(new ProductCommercialDocumentation());
        }

        return $this->commercialDocumentations;
    }

    /**
     * {@inheritdoc}
     */
    public function hasCommercialDocumentations(): bool
    {
        return !$this->commercialDocumentations->isEmpty();
    }

    /**
     * {@inheritdoc}
     */
    public function hasCommercialDocumentation(ImageInterface $image): bool
    {
        return $this->commercialDocumentations->contains($image);
    }

    /**
     * {@inheritdoc}
     */
    public function addCommercialDocumentation(ImageInterface $image): void
    {
        $image->setOwner($this);
        $this->commercialDocumentations->add($image);
    }

    /**
     * {@inheritdoc}
     */
    public function removeCommercialDocumentation(ImageInterface $image): void
    {
        if ($this->hasCommercialDocumentation($image)) {
            $image->setOwner(null);
            $this->commercialDocumentations->removeElement($image);
        }
    }

    public function resetCommercialDocumentations(): void
    {
        $this->commercialDocumentations->clear();
    }

    public function getTechnicalDocumentations(): Collection
    {
        if (0 === $this->technicalDocumentations->count()) {
            $this->addTechnicalDocumentation(new ProductTechnicalDocumentation());
        }

        return $this->technicalDocumentations;
    }

    /**
     * {@inheritdoc}
     */
    public function hasTechnicalDocumentations(): bool
    {
        return !$this->technicalDocumentations->isEmpty();
    }

    /**
     * {@inheritdoc}
     */
    public function hasTechnicalDocumentation(ImageInterface $image): bool
    {
        return $this->technicalDocumentations->contains($image);
    }

    /**
     * {@inheritdoc}
     */
    public function addTechnicalDocumentation(ImageInterface $image): void
    {
        $image->setOwner($this);
        $this->technicalDocumentations->add($image);
    }

    /**
     * {@inheritdoc}
     */
    public function removeTechnicalDocumentation(ImageInterface $image): void
    {
        if ($this->hasTechnicalDocumentation($image)) {
            $image->setOwner(null);
            $this->technicalDocumentations->removeElement($image);
        }
    }

    public function resetTechnicalDocumentations(): void
    {
        $this->technicalDocumentations->clear();
    }

    public function getTechnicalBursts(): Collection
    {
        if (0 === $this->technicalBursts->count()) {
            $this->addTechnicalBurst(new ProductTechnicalBurst());
        }

        return $this->technicalBursts;
    }

    /**
     * {@inheritdoc}
     */
    public function hasTechnicalBursts(): bool
    {
        return !$this->technicalBursts->isEmpty();
    }

    /**
     * {@inheritdoc}
     */
    public function hasTechnicalBurst(ImageInterface $image): bool
    {
        return $this->technicalBursts->contains($image);
    }

    /**
     * {@inheritdoc}
     */
    public function addTechnicalBurst(ImageInterface $image): void
    {
        $image->setOwner($this);
        $this->technicalBursts->add($image);
    }

    /**
     * {@inheritdoc}
     */
    public function removeTechnicalBurst(ImageInterface $image): void
    {
        if ($this->hasTechnicalBurst($image)) {
            $image->setOwner(null);
            $this->technicalBursts->removeElement($image);
        }
    }

    public function resetTechnicalBursts(): void
    {
        $this->technicalBursts->clear();
    }

    public function getProductAssociation(string $type): ?ProductAssociation
    {
        $associations = $this->getAssociations()->filter(function (ProductAssociation $productAssociation) use ($type) {
            return $type === $productAssociation->getType()->getCode();
        });

        return 0 < $associations->count()
            ? $associations->first()
            : null;
    }

    public function getReferenceDisplay(): ?string
    {
        return !empty($this->getManufacturerReference())
            ? $this->getManufacturerReference()
            : !empty($this->getThsReferenceSage())
            ? $this->getThsReferenceSage()
            : null;
    }

    public function isImported(): bool
    {
        return $this->imported;
    }

    /**
     * @return Product
     */
    public function setImported(bool $imported): self
    {
        $this->imported = $imported;

        return $this;
    }

    /**
     * @return Product
     */
    public function activate(): self
    {
        $this->setOnSale(true);
        $this->setDisplayPrice(true);
        $this->setEnabled(true);
        $this->setNoLongerManufactured(false);

        return $this;
    }

    /**
     * @return Product
     */
    public function deactivate(): self
    {
        $this->setOnSale(false);
        $this->setDisplayPrice(false);
        $this->setEnabled(false);
        $this->setNoLongerManufactured(true);

        return $this;
    }

    /**
     * @return Product
     */
    public function clearAttributes(): self
    {
        $this->attributes = new ArrayCollection();

        return $this;
    }

    public function getTitle(): ?string
    {
        return $this->getTranslation()->getTitle();
    }

    public function setTitle(?string $title): void
    {
        $this->getTranslation()->setTitle($title);
    }

    /**
     * {@inheritdoc}
     */
    protected function createTranslation(): ProductTranslationInterface
    {
        return new ProductTranslation();
    }

    public function isDisplaySearch(): bool
    {
        return $this->displaySearch;
    }

    public function setDisplaySearch(bool $displaySearch): Product
    {
        $this->displaySearch = $displaySearch;

        return $this;
    }

    public function getUrl(): string
    {
        return $this->url;
    }

    public function setUrl(string $url): Product
    {
        $this->url = $url;

        return $this;
    }
}
