<?php

declare(strict_types=1);

namespace App\Form\Extension;

use App\Entity\Address;
use App\Entity\Customer;
use Sylius\Bundle\AddressingBundle\Form\Type\AddressType;
use Symfony\Component\Form\AbstractTypeExtension;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Validator\Constraints\NotBlank;

final class AddressTypeExtension extends AbstractTypeExtension
{
    /**
     * {@inheritdoc}
     */
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('type', ChoiceType::class, [
                'choices' => [
                    Customer::TYPE_PARTICULIER => Customer::TYPE_PARTICULIER,
                    Customer::TYPE_SOCIETE => Customer::TYPE_SOCIETE,
                    Customer::TYPE_ADMINISTRATION => Customer::TYPE_ADMINISTRATION,
                ],
            ])
            ->add('civility', ChoiceType::class, [
                'choices' => [
                    Customer::CIVILITY_MR => Customer::CIVILITY_MR,
                    Customer::CIVILITY_MME => Customer::CIVILITY_MME,
                ],
            ])
            ->add('firstName', TextType::class, [
                'label' => null,
                'attr' => [
                    'placeholder' => 'sylius.form.address.first_name',
                ],
            ])
            ->add('lastName', TextType::class, [
                'label' => 'sylius.form.address.last_name',
                'attr' => [
                    'placeholder' => 'sylius.form.address.last_name',
                ],
            ])
            ->add('phoneNumber', TextType::class, [
                'required' => false,
                'label' => 'sylius.form.address.phone_number',
                'attr' => [
                    'placeholder' => 'sylius.form.address.phone_number',
                ],
            ])
            ->add('mobilePhoneNumber', TextType::class, [
                'required' => false,
                'label' => 'sylius.form.address.mobile_phone_number',
                'attr' => [
                    'placeholder' => 'sylius.form.address.mobile_phone_number',
                ],
            ])
            ->add('company', TextType::class, [
                'required' => false,
                'label' => 'app.ui.form.address.company',
                'attr' => [
                    'placeholder' => 'app.ui.form.address.company',
                ],
            ])
            ->add('street', TextType::class, [
                'label' => 'sylius.form.address.street',
                'attr' => [
                    'placeholder' => 'sylius.form.address.street',
                ],
            ])
            ->add('city', TextType::class, [
                'label' => 'sylius.form.address.city',
                'attr' => [
                    'placeholder' => 'sylius.form.address.city',
                ],
            ])
            ->add('postcode', TextType::class, [
                'label' => 'sylius.form.address.postcode',
                'attr' => [
                    'placeholder' => 'sylius.form.address.postcode',
                ],
            ])
        ;

        $builder->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event): void {
            /** @var Address $data */
            $data = $event->getData();

            if ('Particulier' === $data['type']) {
                $event->getForm()
                    ->add('civility', ChoiceType::class, [
                        'choices' => $civilities = [
                            Customer::CIVILITY_MR => Customer::CIVILITY_MR,
                            Customer::CIVILITY_MME => Customer::CIVILITY_MME,
                        ],
                    ]);
            } elseif ('Société' === $data['type']) {
                $event->getForm()
                    ->add('civility', ChoiceType::class, [
                        'choices' => [
                            'Autre' => Customer::CIVILITY_SOCIETE,
                            Customer::CIVILITY_EARL => Customer::CIVILITY_EARL,
                            Customer::CIVILITY_EURL => Customer::CIVILITY_EURL,
                            Customer::CIVILITY_GAEC => Customer::CIVILITY_GAEC,
                            Customer::CIVILITY_SARL => Customer::CIVILITY_SARL,
                            Customer::CIVILITY_SA => Customer::CIVILITY_SA,
                            Customer::CIVILITY_SAS => Customer::CIVILITY_SAS,
                            Customer::CIVILITY_SC => Customer::CIVILITY_SC,
                            Customer::CIVILITY_SCA => Customer::CIVILITY_SCA,
                            Customer::CIVILITY_SCI => Customer::CIVILITY_SCI,
                            Customer::CIVILITY_SCOP => Customer::CIVILITY_SCOP,
                            Customer::CIVILITY_SCP => Customer::CIVILITY_SCP,
                            Customer::CIVILITY_SICA => Customer::CIVILITY_SICA,
                            Customer::CIVILITY_SNC => Customer::CIVILITY_SNC,
                        ],
                    ]);
            } else {
                $data['civility'] = '';
                $event->setData($data);
                $event->getForm()
                    ->add('civility', ChoiceType::class, [
                        'choices' => ['' => ''],
                        'required' => false,
                    ]);
            }

            if ('Particulier' !== $data['type']) {
                $event->getForm()
                    ->add('company', TextType::class, [
                        'required' => true,
                        'label' => 'app.ui.form.address.company',
                        'attr' => [
                            'placeholder' => 'app.ui.form.address.company',
                        ],
                        'constraints' => [
                            new NotBlank([
                                'message' => 'sylius.address.company.not_blank',
                                'groups' => ['app_b2b', 'app_b2g'],
                            ]),
                        ],
                    ]);
            }
        });
    }

    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults([
            'validation_groups' => function (FormInterface $form) {
                $data = $form->getData();

                if ('Société' === $data->getType()) {
                    return ['app_b2b', 'sylius'];
                } elseif ('Administration' === $data->getType()) {
                    return ['app_b2g', 'sylius'];
                } else {
                    return ['app_b2c', 'sylius'];
                }
            },
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public static function getExtendedTypes(): iterable
    {
        return [AddressType::class];
    }
}
