<?php

declare(strict_types=1);

namespace App\Importer;

use App\Entity\Adjustment;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\AdjustmentInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\OrderItem;
use Sylius\Component\Core\Model\OrderItemUnit;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Order\Modifier\OrderItemQuantityModifierInterface;
use Sylius\Component\Product\Generator\SlugGeneratorInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;

/**
 * Class OrderLineImporter.
SELECT ca.id_commande, ca.id_article, nhp.modele,
nhp.reference_ths, ca.quantite, ca.`prix_unitaire_ht`,
ca.`prix_total_ht`, ca.`prix_total_ttc`
FROM `commande_article` ca
LEFT JOIN nettoyeur_haute_pression nhp on ca.id_nettoyeur = nhp.id_nettoyeur
 */
class OrderLineImporter implements ImporterInterface
{
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    /**
     * @var FactoryInterface
     */
    private $adjustmentFactory;

    /**
     * @var FactoryInterface
     */
    private $orderItemFactory;

    /**
     * @var FactoryInterface
     */
    private $orderItemUnitFactory;

    /**
     * @var RepositoryInterface
     */
    private $orderRepository;

    /**
     * @var RepositoryInterface
     */
    private $productRepository;

    /**
     * @var OrderItemQuantityModifierInterface
     */
    private $orderItemQuantityModifier;

    /**
     * @var SlugGeneratorInterface
     */
    private $slugGenerator;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * OrderImporter constructor.
     *
     * @param EntityManagerInterface             $entityManager
     * @param FactoryInterface                   $adjustmentFactory
     * @param FactoryInterface                   $orderItemFactory
     * @param FactoryInterface                   $orderItemUnitFactory
     * @param RepositoryInterface                $orderRepository
     * @param RepositoryInterface                $productRepository
     * @param OrderItemQuantityModifierInterface $orderItemQuantityModifier
     * @param SlugGeneratorInterface             $slugGenerator
     * @param LoggerInterface                    $logger
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        FactoryInterface $adjustmentFactory,
        FactoryInterface $orderItemFactory,
        FactoryInterface $orderItemUnitFactory,
        RepositoryInterface $orderRepository,
        RepositoryInterface $productRepository,
        OrderItemQuantityModifierInterface $orderItemQuantityModifier,
        SlugGeneratorInterface $slugGenerator,
        LoggerInterface $logger
    ) {
        $this->entityManager = $entityManager;
        $this->adjustmentFactory = $adjustmentFactory;
        $this->orderItemFactory = $orderItemFactory;
        $this->orderItemUnitFactory = $orderItemUnitFactory;
        $this->orderRepository = $orderRepository;
        $this->productRepository = $productRepository;
        $this->orderItemQuantityModifier = $orderItemQuantityModifier;
        $this->slugGenerator = $slugGenerator;
        $this->logger = $logger;
    }

    /**
     * Import order line from THS V1.
     *
     * @param array $line
     */
    public function import(array $line): void
    {
        $order = $this->getOrder((int) $line[0]);

        if (!$order) {
            $this->logger->warning('Order not exist', [$line[0]]);

            return;
        }

        $code = 'NULL' != $line[1]
            ? $line[1]
            : $this->slugGenerator->generate($line[2].' '.$line[3]);
        $product = $this->getProduct($code);

        if (!$product) {
            $this->logger->warning('Product not exist', [$code]);

            return;
        }

        $productVariant = $product->getVariants()->first();

        /** @var OrderItem $orderItem */
        $orderItem = $this->orderItemFactory->createNew();
        $orderItem->setOrder($order);
        $orderItem->setVariant($productVariant);
        $price = 'NULL' !== $line[5] ? $line[5] : 0;
        $amount = explode('.', (string) ($price * 100));
        $orderItem->setUnitPrice((int) $amount[0]);
        $orderItem->setProductName($product->getName());
        $orderItem->setVariantName($productVariant->getName());

        $vatValue = explode('.', (string) (round(($line[7] - $line[6]) / $line[4], 2) * 100));

        for ($i = 0; $i < $line[4]; ++$i) {
            /** @var OrderItemUnit $orderItemUnit */
            $orderItemUnit = $this->orderItemUnitFactory->createForItem($orderItem);
            $orderItemUnit->setCreatedAt($order->getCreatedAt());
            $orderItemUnit->setUpdatedAt($order->getUpdatedAt());

            // Tax
            /** @var Adjustment $adjustment */
            $adjustment = $this->adjustmentFactory->createNew();
            $adjustment->setType(AdjustmentInterface::TAX_ADJUSTMENT);
            $adjustment->setLabel('TVA');
            $adjustment->setAmount((int) $vatValue[0]);
            $adjustment->setNeutral(false);

            $orderItemUnit->addAdjustment($adjustment);

            $orderItem->addUnit($orderItemUnit);
        }

        $this->entityManager->persist($orderItem);
    }

    /**
     * @param int $id
     *
     * @return OrderInterface|null
     */
    private function getOrder(int $id): ?OrderInterface
    {
        return $this->orderRepository->findOneBy(['oldOrderId' => $id]);
    }

    /**
     * @param string $code
     *
     * @return ProductInterface|null
     */
    private function getProduct(string $code): ?ProductInterface
    {
        return $this->productRepository->findOneBy(['code' => $code]);
    }
}
