<?php

declare(strict_types=1);

namespace App\Importer;

use App\Entity\Product;
use App\Entity\ProductCommercialDocumentation;
use App\Entity\ProductTechnicalBurst;
use App\Entity\ProductTechnicalDocumentation;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\LoggerInterface;
use Sylius\Component\Attribute\Model\AttributeValueInterface;
use Sylius\Component\Core\Model\Channel;
use Sylius\Component\Core\Model\ChannelPricing;
use Sylius\Component\Core\Model\ProductTaxon;
use Sylius\Component\Core\Model\ProductVariant;
use Sylius\Component\Core\Uploader\ImageUploaderInterface;
use Sylius\Component\Product\Generator\SlugGeneratorInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Sylius\Component\Shipping\Model\ShippingCategory;
use Sylius\Component\Taxation\Model\TaxCategory;
use Sylius\Component\Taxonomy\Model\Taxon;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * Class PressureWasherImporter.
 *
 * Query: SELECT `id_nettoyeur`, m.libelle, `modele`, `poste_fixe`,
`groupe_moteur_pompe`, `eau_froide`, `eau_chaude`, `vapeur`, `temperature`,
`pression_service`, `debit_litre_min`, `debit_litre_heure`, `calibre`,
`electrique_220`, `electrique_220_a`, `electrique_220_kw`,
`electrique_220_hz`, `electrique_380`, `electrique_380_a`,
`electrique_380_kw`, `electrique_380_hz`, `moteur_thermique`,
`moteur_thermique_info`, `poids`, `description`, `reference_ths`,
`reference_constructeur`, `prix_vente_ht`, `taux_remise_fournisseur`,
`prix_vente_ths`, `marge_pourcentage`, `marge_euro`, `prix_ttc`, `stock`,
`franco_de_port`, `promo`, `url_article`, `balise_title`, `balise_meta`,
`photo_1`, `photo_2`, `photo_3`, `photo_4`, `photo_5`, `video_flv`,
`doc_commercial`, `doc_technique`, `doc_technique_espace_pro`,
`eclate_technique`, `eclate_technique_espace_pro`, `actif`,
`disponible_vente`, `visible_moteur_recherche`, `afficher_prix`,
`plus_fabriquer`, `no_index`, `vente_pieces_detachees` FROM
`nettoyeur_haute_pression` nhp
INNER JOIN `marque` m ON m.id_marque = nhp.id_marque
 */
class PressureWasherImporter implements ImporterInterface
{
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    /**
     * @var FactoryInterface
     */
    private $attributeValueFactory;
    /**
     * @var FactoryInterface
     */
    private $channelPricingFactory;

    /**
     * @var FactoryInterface
     */
    private $productFactory;

    /**
     * @var FactoryInterface
     */
    private $productImageFactory;

    /**
     * @var FactoryInterface
     */
    private $productVariantFactory;

    /**
     * @var RepositoryInterface
     */
    private $productTaxonFactory;

    /**
     * @var RepositoryInterface
     */
    private $channelRepository;

    /**
     * @var RepositoryInterface
     */
    private $productAttributeRepository;

    /**
     * @var RepositoryInterface
     */
    private $productRepository;

    /**
     * @var RepositoryInterface
     */
    private $shippingCategoryRepository;

    /**
     * @var RepositoryInterface
     */
    private $taxCategoryRepository;

    /**
     * @var RepositoryInterface
     */
    private $taxonRepository;

    /**
     * @var SlugGeneratorInterface
     */
    private $slugGenerator;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ImageUploaderInterface
     */
    private $imageUploader;

    /**
     * PumpImporter constructor.
     *
     * @param EntityManagerInterface $entityManager
     * @param FactoryInterface       $channelPricingFactory
     * @param FactoryInterface       $attributeValueFactory
     * @param FactoryInterface       $productFactory
     * @param FactoryInterface       $productImageFactory
     * @param FactoryInterface       $productTaxonFactory
     * @param FactoryInterface       $productVariantFactory
     * @param RepositoryInterface    $channelRepository
     * @param RepositoryInterface    $productAttributeRepository
     * @param RepositoryInterface    $productRepository
     * @param RepositoryInterface    $shippingCategoryRepository
     * @param RepositoryInterface    $taxCategoryRepository
     * @param RepositoryInterface    $taxonRepository
     * @param SlugGeneratorInterface $slugGenerator
     * @param LoggerInterface        $logger
     * @param ImageUploaderInterface $imageUploader
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        FactoryInterface $channelPricingFactory,
        FactoryInterface $attributeValueFactory,
        FactoryInterface $productFactory,
        FactoryInterface $productImageFactory,
        FactoryInterface $productTaxonFactory,
        FactoryInterface $productVariantFactory,
        RepositoryInterface $channelRepository,
        RepositoryInterface $productAttributeRepository,
        RepositoryInterface $productRepository,
        RepositoryInterface $shippingCategoryRepository,
        RepositoryInterface $taxCategoryRepository,
        RepositoryInterface $taxonRepository,
        SlugGeneratorInterface $slugGenerator,
        LoggerInterface $logger,
        ImageUploaderInterface $imageUploader
    ) {
        $this->entityManager = $entityManager;
        $this->channelPricingFactory = $channelPricingFactory;
        $this->attributeValueFactory = $attributeValueFactory;
        $this->productFactory = $productFactory;
        $this->productImageFactory = $productImageFactory;
        $this->productTaxonFactory = $productTaxonFactory;
        $this->productVariantFactory = $productVariantFactory;
        $this->channelRepository = $channelRepository;
        $this->productAttributeRepository = $productAttributeRepository;
        $this->productRepository = $productRepository;
        $this->shippingCategoryRepository = $shippingCategoryRepository;
        $this->taxCategoryRepository = $taxCategoryRepository;
        $this->taxonRepository = $taxonRepository;
        $this->slugGenerator = $slugGenerator;
        $this->logger = $logger;
        $this->imageUploader = $imageUploader;
    }

    /**
     * Import pressure washer from THS V1.
     *
     * @param array $line
     */
    public function import(array $line): void
    {
        $code = mb_strtoupper($this->slugGenerator->generate($line[2].' '.$line[25]));
        $this->logger->info('Process', [$code]);
        $product = $this->productRepository->findOneBy(['code' => $code]);

        if ($product) {
            $this->logger->warning('Reference THS already exist', [$code]);
        } else {
            /** @var Product $product */
            $product = $this->productFactory->createNew();

            $product->setName($line[2]);
            $product->setCode(strtoupper($code));
            $product->setManufacturerReference($line[26]);
            $product->setMetaDescription($line[38]);
            $product->setEnabled('1' === $line[50]);
            $product->addChannel($this->getChannel());
            $product->setSlug($this->slugGenerator->generate(mb_strtolower($line[1].' '.$line[2])));
            $product->setDescription($line[24]);
            $product->setFreePort('1' === $line[34]);
            $product->setOnSale('1' == $line[51]);
            $product->setDisplayMenu('1' == $line[52]);
            $product->setDisplayPrice('1' == $line[53]);
            $product->setNoLongerManufactured('1' == $line[54]);
            $product->setNoIndex('1' == $line[55]);
            $product->setSaleSpareParts('1' == $line[56]);

            /**
             * Taxons.
             */
            $taxon = $this->getTaxon($line[1]);
            $product->setMainTaxon($taxon);

            /** @var ProductTaxon $productTaxon */
            $productTaxon = $this->productTaxonFactory->createNew();
            $productTaxon->setTaxon($taxon);
            $productTaxon->setProduct($product);
            $product->addProductTaxon($productTaxon);

            $taxon = $this->getMainTaxon('NHP');

            /** @var ProductTaxon $productTaxon */
            $productTaxon = $this->productTaxonFactory->createNew();
            $productTaxon->setTaxon($taxon);
            $productTaxon->setProduct($product);
            $product->addProductTaxon($productTaxon);

            /**
             * Attributes.
             */
            $product = $this->addAttributes($product, 'pressure', $line[9]);
            $product = $this->addAttributes($product, 'flow_liters_minute', $line[10]);
            $product = $this->addAttributes($product, 'flow_liters_hour', $line[11]);
            $product = $this->addAttributes($product, 'cold_water', '1' === $line[5]);
            $product = $this->addAttributes($product, 'hot_water', '1' === $line[6]);
            $product = $this->addAttributes($product, 'steam', '1' === $line[7]);
            $product = $this->addAttributes($product, 'temperature', $line[8]);
            $product = $this->addAttributes($product, 'fixed_station', '1' === $line[3]);
            $product = $this->addAttributes($product, 'pump_motor_group', '1' === $line[4]);
            $product = $this->addAttributes($product, 'thermal_motor', '1' === $line[21]);
            $product = $this->addAttributes($product, 'thermal_motor_information', $line[22]);

            /** @var ProductVariant $productVariant */
            $productVariant = $this->productVariantFactory->createNew();

            $productVariant->setName($line[2]);
            $productVariant->setCode($code);
            $productVariant->setPosition(1);
            $productVariant->setProduct($product);
            $productVariant->setWeight((float) $line[23]);
            $productVariant->setTaxCategory($this->getTaxCategory());
            $productVariant->setShippingCategory($this->getShippingCategory('OUTSIZE'));

            /** @var ChannelPricing $channelPricing */
            $channelPricing = $this->channelPricingFactory->createNew();
            $channelPricing->setChannelCode($this->getChannel()->getCode());
            $channelPricing->setProductVariant($productVariant);
            $channelPricing->setPrice((int) $line[29] * 100);
            $channelPricing->setOriginalPrice((int) $line[29] * 100);

            $productVariant->addChannelPricing($channelPricing);
            $product->addVariant($productVariant);

            foreach ([39, 40, 41, 42, 43] as $i) {
                $imagePath = __DIR__."/../../../old_images/nettoyeurs/{$line[$i]}";

                if ('' != $line[$i] && 'NULL' != $line[$i] && file_exists($imagePath)) {
                    $uploadedImage = new UploadedFile($imagePath, basename($imagePath));

                    /** @var ImageInterface $productImage */
                    $productImage = $this->productImageFactory->createNew();
                    $productImage->setFile($uploadedImage);
                    $productImage->setType($line[0]);
                    $this->imageUploader->upload($productImage);
                    $product->addImage($productImage);
                }
            }
        }

        $product->resetCommercialDocumentations();
        $product->resetTechnicalBursts();
        $product->resetTechnicalDocumentations();

        $imagePath = __DIR__."/../../../pdf/1/1/{$line[46]}";

        if ('' != $line[46] && 'NULL' != $line[46] && file_exists($imagePath)) {
            $uploadedImage = new UploadedFile($imagePath, basename($imagePath));

            $doc = new ProductTechnicalDocumentation();
            $doc->setActive(true);
            $doc->setProfessionalPlace('1' === $line[47]);
            $doc->setFile($uploadedImage);
            $doc->setType('PDF');
            $this->imageUploader->upload($doc);
            $product->addTechnicalDocumentation($doc);
        }

        $imagePath = __DIR__."/../../../pdf/1/3/{$line[45]}";

        if ('' != $line[45] && 'NULL' != $line[45] && file_exists($imagePath)) {
            $uploadedImage = new UploadedFile($imagePath, basename($imagePath));

            $doc = new ProductCommercialDocumentation();
            $doc->setActive(true);
            $doc->setFile($uploadedImage);
            $doc->setType('PDF');
            $this->imageUploader->upload($doc);
            $product->addCommercialDocumentation($doc);
        }

        $imagePath = __DIR__."/../../../pdf/1/2/{$line[48]}";

        if ('' != $line[48] && 'NULL' != $line[48] && file_exists($imagePath)) {
            $uploadedImage = new UploadedFile($imagePath, basename($imagePath));

            $doc = new ProductTechnicalBurst();
            $doc->setActive(true);
            $doc->setProfessionalPlace('1' === $line[49]);
            $doc->setFile($uploadedImage);
            $doc->setType('PDF');
            $this->imageUploader->upload($doc);
            $product->addTechnicalBurst($doc);
        }

        $this->entityManager->persist($product);
    }

    /**
     * @param Product     $product
     * @param string      $code
     * @param string|bool $value
     *
     * @return Product
     */
    private function addAttributes(Product $product, string $code, $value): Product
    {
        /** @var AttributeValueInterface $attributeValue */
        $attributeValue = $this->attributeValueFactory->createNew();
        $attribute = $this->productAttributeRepository->findOneBy(['code' => $code]);
        $attributeValue->setAttribute($attribute);
        $attributeValue->setValue($value);
        $attributeValue->setLocaleCode('fr_FR');
        $product->addAttribute($attributeValue);

        return $product;
    }

    /**
     * @param string $code
     *
     * @return Taxon
     */
    private function getMainTaxon(string $code): Taxon
    {
        return $this->taxonRepository->findOneBy(['code' => $code]);
    }

    /**
     * @param string $code
     *
     * @return Taxon
     */
    private function getTaxon(string $code): Taxon
    {
        return $this->taxonRepository->findOneBy(['code' => 'N_'.str_replace([' ', '/'], '', trim(strtoupper($code)))]);
    }

    /**
     * @return Channel
     */
    private function getChannel(): Channel
    {
        return $this->channelRepository->findOneBy(['code' => 'ECOMMERCE']);
    }

    /**
     * @return TaxCategory
     */
    private function getTaxCategory(): TaxCategory
    {
        return $this->taxCategoryRepository->findOneBy(['code' => 'SALE20']);
    }

    /**
     * @param string $code
     *
     * @return ShippingCategory
     */
    private function getShippingCategory(string $code): ShippingCategory
    {
        return $this->shippingCategoryRepository->findOneBy(['code' => $code]);
    }
}
