<?php

declare(strict_types=1);

namespace App\Importer;

use App\Entity\Product;
use App\Entity\ProductCommercialDocumentation;
use App\Entity\ProductTechnicalBurst;
use App\Entity\ProductTechnicalDocumentation;
use Doctrine\ORM\EntityManagerInterface;
use Sylius\Component\Attribute\Model\AttributeValueInterface;
use Sylius\Component\Core\Model\Channel;
use Sylius\Component\Core\Model\ProductTaxon;
use Sylius\Component\Core\Uploader\ImageUploaderInterface;
use Sylius\Component\Product\Generator\SlugGeneratorInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Sylius\Component\Taxonomy\Model\Taxon;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * Class PumpImporter.
 *
 * Query: SELECT `id_pompe`, m.libelle, `modele`, `serie`, `pression`, `debit_litre_min`,
`debit_litre_heure`, `vitesse_rotation`, `puissance`, `poids`, `description`,
`url_article`, `balise_title`, `balise_meta`, `photo_1`, `doc_commercial`,
`doc_technique`, `doc_technique_espace_pro`, `eclate_technique`, `eclate_technique_espace_pro`, `actif`
FROM `pompe` p
INNER JOIN `marque` m on m.id_marque = p.id_marque
 */
class PumpImporter implements ImporterInterface
{
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    /**
     * @var FactoryInterface
     */
    private $attributeValueFactory;

    /**
     * @var FactoryInterface
     */
    private $productFactory;

    /**
     * @var FactoryInterface
     */
    private $productImageFactory;

    /**
     * @var RepositoryInterface
     */
    private $productTaxonFactory;

    /**
     * @var RepositoryInterface
     */
    private $channelRepository;

    /** @var RepositoryInterface */
    private $productRepository;

    /**
     * @var RepositoryInterface
     */
    private $productAttributeRepository;

    /**
     * @var RepositoryInterface
     */
    private $taxonRepository;

    /**
     * @var SlugGeneratorInterface
     */
    private $slugGenerator;

    /**
     * @var ImageUploaderInterface
     */
    private $imageUploader;

    /**
     * PumpImporter constructor.
     *
     * @param EntityManagerInterface $entityManager
     * @param FactoryInterface       $attributeValueFactory
     * @param FactoryInterface       $productFactory
     * @param FactoryInterface       $productImageFactory
     * @param FactoryInterface       $productTaxonFactory
     * @param RepositoryInterface    $channelRepository
     * @param RepositoryInterface    $productRepository
     * @param RepositoryInterface    $productAttributeRepository
     * @param RepositoryInterface    $taxonRepository
     * @param SlugGeneratorInterface $slugGenerator
     * @param ImageUploaderInterface $imageUploader
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        FactoryInterface $attributeValueFactory,
        FactoryInterface $productFactory,
        FactoryInterface $productImageFactory,
        FactoryInterface $productTaxonFactory,
        RepositoryInterface $channelRepository,
        RepositoryInterface $productRepository,
        RepositoryInterface $productAttributeRepository,
        RepositoryInterface $taxonRepository,
        SlugGeneratorInterface $slugGenerator,
        ImageUploaderInterface $imageUploader
    ) {
        $this->entityManager = $entityManager;
        $this->attributeValueFactory = $attributeValueFactory;
        $this->productFactory = $productFactory;
        $this->productImageFactory = $productImageFactory;
        $this->productTaxonFactory = $productTaxonFactory;
        $this->channelRepository = $channelRepository;
        $this->productRepository = $productRepository;
        $this->productAttributeRepository = $productAttributeRepository;
        $this->taxonRepository = $taxonRepository;
        $this->slugGenerator = $slugGenerator;
        $this->imageUploader = $imageUploader;
    }

    /**
     * Import pump from THS V1.
     *
     * @param array $line
     */
    public function import(array $line): void
    {
        $code = mb_strtoupper($line[2]);
        $product = $this->productRepository->findOneBy(['code' => $code]);

        if (!$product) {
            /** @var Product $product */
            $product = $this->productFactory->createNew();

            $product->setName($line[2]);
            $product->setCode($code);
            $product->setMetaDescription($line[13]);
            $product->setEnabled('1' === $line[20]);
            $product->addChannel($this->getChannel());
            $product->setSlug($this->slugGenerator->generate(mb_strtolower($line[1].' '.$line[2])));

            /**
             * Taxons.
             */
            $taxon = $this->getTaxon($line[1]);
            $product->setMainTaxon($taxon);

            /** @var ProductTaxon $productTaxon */
            $productTaxon = $this->productTaxonFactory->createNew();
            $productTaxon->setTaxon($taxon);
            $productTaxon->setProduct($product);
            $product->addProductTaxon($productTaxon);

            $taxon = $this->getMainTaxon('POMPE');

            /** @var ProductTaxon $productTaxon */
            $productTaxon = $this->productTaxonFactory->createNew();
            $productTaxon->setTaxon($taxon);
            $productTaxon->setProduct($product);
            $product->addProductTaxon($productTaxon);

            /**
             * Attributes.
             */
            $product = $this->addAttributes($product, 'pressure', $line[4]);
            $product = $this->addAttributes($product, 'flow_liters_minute', $line[5]);
            $product = $this->addAttributes($product, 'flow_liters_hour', $line[6]);
            $product = $this->addAttributes($product, 'rotation_speed', $line[7]);

            $imagePath = __DIR__."/../../../old_images/pompes/{$line[14]}";

            if ('' != $line[14] && 'NULL' != $line[14] && file_exists($imagePath)) {
                $uploadedImage = new UploadedFile($imagePath, basename($imagePath));

                /** @var ImageInterface $productImage */
                $productImage = $this->productImageFactory->createNew();
                $productImage->setFile($uploadedImage);
                $productImage->setType($line[0]);
                $this->imageUploader->upload($productImage);
                $product->addImage($productImage);
            }
        }

        $product->resetCommercialDocumentations();
        $product->resetTechnicalBursts();
        $product->resetTechnicalDocumentations();

        $imagePath = __DIR__."/../../../pdf/2/2/{$line[16]}";

        if ('' != $line[16] && 'NULL' != $line[16] && file_exists($imagePath)) {
            $uploadedImage = new UploadedFile($imagePath, basename($imagePath));

            $doc = new ProductTechnicalDocumentation();
            $doc->setActive(true);
            $doc->setProfessionalPlace('1' === $line[17]);
            $doc->setFile($uploadedImage);
            $doc->setType('PDF');
            $this->imageUploader->upload($doc);
            $product->addTechnicalDocumentation($doc);
        }

        $imagePath = __DIR__."/../../../pdf/2/3/{$line[15]}";

        if ('' != $line[15] && 'NULL' != $line[15] && file_exists($imagePath)) {
            $uploadedImage = new UploadedFile($imagePath, basename($imagePath));

            $doc = new ProductCommercialDocumentation();
            $doc->setActive(true);
            $doc->setFile($uploadedImage);
            $doc->setType('PDF');
            $this->imageUploader->upload($doc);
            $product->addCommercialDocumentation($doc);
        }

        $imagePath = __DIR__."/../../../pdf/2/1/{$line[18]}";

        if ('' != $line[18] && 'NULL' != $line[18] && file_exists($imagePath)) {
            $uploadedImage = new UploadedFile($imagePath, basename($imagePath));

            $doc = new ProductTechnicalBurst();
            $doc->setActive(true);
            $doc->setProfessionalPlace('1' === $line[19]);
            $doc->setFile($uploadedImage);
            $doc->setType('PDF');
            $this->imageUploader->upload($doc);
            $product->addTechnicalBurst($doc);
        }

        $this->entityManager->persist($product);
    }

    /**
     * @param Product $product
     * @param string  $code
     * @param string  $value
     *
     * @return Product
     */
    private function addAttributes(Product $product, string $code, string $value): Product
    {
        /** @var AttributeValueInterface $attributeValue */
        $attributeValue = $this->attributeValueFactory->createNew();
        $attribute = $this->productAttributeRepository->findOneBy(['code' => $code]);
        $attributeValue->setAttribute($attribute);
        $attributeValue->setValue($value);
        $attributeValue->setLocaleCode('fr_FR');
        $product->addAttribute($attributeValue);

        return $product;
    }

    /**
     * @param string $code
     *
     * @return Taxon
     */
    private function getMainTaxon(string $code): Taxon
    {
        return $this->taxonRepository->findOneBy(['code' => $code]);
    }

    /**
     * @param string $code
     *
     * @return Taxon
     */
    private function getTaxon(string $code): Taxon
    {
        return $this->taxonRepository->findOneBy(['code' => 'P_'.str_replace(' ', '', trim(strtoupper($code)))]);
    }

    /**
     * @return Channel
     */
    private function getChannel(): Channel
    {
        return $this->channelRepository->findOneBy(['code' => 'ECOMMERCE']);
    }
}
