<?php

declare(strict_types=1);

namespace App\Importer;

use App\Entity\Address;
use App\Entity\Customer;
use Doctrine\ORM\EntityManagerInterface;
use Sylius\Component\Core\Model\ShopUser;
use Sylius\Component\Resource\Factory\FactoryInterface;

/**
 * Class UserImporter.
 *
 * SELECT c.id_client, c.type, c.civilite, c.nom, c.prenom, u.last_name, u.first_name,
u.email_address, u.is_active, u.last_login, u.created_at, u.updated_at,
c.email, c.adresse, c.adresse_complement, c.code_postal, c.ville, c.pays,
c.telephone, c.fax, c.export_sage, c.raison_sociale, u.salt, u.password
FROM sf_guard_user u
INNER JOIN client c ON u.id = c.id_user
 */
class UserImporter implements ImporterInterface
{
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;

    /**
     * @var FactoryInterface
     */
    private $addressFactory;

    /**
     * @var FactoryInterface
     */
    private $customerFactory;

    /**
     * @var FactoryInterface
     */
    private $shopUserFactory;

    /**
     * PumpImporter constructor.
     *
     * @param EntityManagerInterface $entityManager
     * @param FactoryInterface       $addressFactory
     * @param FactoryInterface       $customerFactory
     * @param FactoryInterface       $shopUserFactory
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        FactoryInterface $addressFactory,
        FactoryInterface $customerFactory,
        FactoryInterface $shopUserFactory
    ) {
        $this->entityManager = $entityManager;
        $this->addressFactory = $addressFactory;
        $this->customerFactory = $customerFactory;
        $this->shopUserFactory = $shopUserFactory;
    }

    /**
     * Import user from THS V1.
     *
     * @param array $line
     */
    public function import(array $line): void
    {
        $email = $line[7];
        $enabled = '1' == $line[8];
        $createdAt = \DateTime::createFromFormat('Y-m-d H:i:s', $line[10]);
        $updatedAt = \DateTime::createFromFormat('Y-m-d H:i:s', $line[11]);
        $phoneNumber = $line[18];

        /** @var Address $address */
        $address = $this->addressFactory->createNew();
        $address->setType($line[1]);
        $address->setCivility($line[2]);
        $address->setLastName($line[5]);
        $address->setFirstName($line[6]);
        $address->setFaxNumber($line[19]);
        $address->setPhoneNumber($phoneNumber);
        $address->setCreatedAt($createdAt);

        if ($updatedAt) {
            $address->setUpdatedAt($updatedAt);
        }

        $address->setStreet($line[13].('' != $line[14] ? ' '.$line[14] : ''));
        $address->setCity($line[16]);
        $address->setPostcode($line[15]);
        // Set to FR => $line[17]
        $address->setCountryCode('FR');

        /** @var Customer $customer */
        $customer = $this->customerFactory->createNew();
        $customer->setEmail($email);
        $customer->setEmailCanonical($email);
        $customer->setCreatedAt($createdAt);

        if ($updatedAt) {
            $customer->setUpdatedAt($updatedAt);
        }

        $customer->setLastName($line[3]);
        $customer->setFirstName($line[4]);
        $customer->setPhoneNumber($phoneNumber);
        $customer->setOldCustomerId((int) $line[0]);

        if ($line[18]) {
            $customer->setFaxNumber($line[19]);
        }

        $customer->setExportSage('1' == $line[20]);

        if ('NULL' != $line[21]) {
            $customer->setCompanyName($line[21]);
        }

        $type = $line[1];

        if ('NULL' == $type && in_array($line[2], [Customer::CIVILITY_MR, Customer::CIVILITY_MME])) {
            $type = Customer::TYPE_PARTICULIER;
        } else {
            $type = Customer::TYPE_SOCIETE;
        }

        $customer->setType($type);
        $customer->setCivility($line[2]);
        $customer->setSubscribedToNewsletter(false);
        $customer->setDefaultAddress($address);
        $customer->addAddress($address);

        /** @var ShopUser $shopUser */
        $shopUser = $this->shopUserFactory->createNew();
        $shopUser->setCustomer($customer);
        $shopUser->setUsername($email);
        $shopUser->setEmail($email);
        $shopUser->setEmailCanonical($email);
        $shopUser->setEnabled($enabled);
        $shopUser->setCreatedAt($createdAt);
        $shopUser->setPassword($line[23]);
        $shopUser->setSalt($line[22]);
        $shopUser->setEncoderName('ths');

        if ($updatedAt) {
            $shopUser->setUpdatedAt($updatedAt);
        }

        $shopUser->setLastLogin('NULL' == $line[9] ? null : \DateTime::createFromFormat('Y-m-d H:i:s', $line[9]));

        $this->entityManager->persist($shopUser);
    }
}
