<?php

declare(strict_types=1);

namespace App\OrderProcessing;

use App\Entity\Adjustment;
use App\Entity\Order;
use App\Entity\Setting;
use Sylius\Component\Order\Model\OrderInterface as BaseOrderInterface;
use Sylius\Component\Order\Processor\OrderProcessorInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;

final class OrderManagementFeesProcessor implements OrderProcessorInterface
{
    /** @var RepositoryInterface */
    private $repositorySetting;

    /** @var FactoryInterface */
    private $factoryAdjustment;

    /**
     * OrderManagementFeesProcessor constructor.
     *
     * @param RepositoryInterface $repositorySetting
     * @param FactoryInterface    $factoryAdjustment
     */
    public function __construct(RepositoryInterface $repositorySetting, FactoryInterface $factoryAdjustment)
    {
        $this->repositorySetting = $repositorySetting;
        $this->factoryAdjustment = $factoryAdjustment;
    }

    /**
     * {@inheritdoc}
     */
    public function process(BaseOrderInterface $order): void
    {
        if (in_array($order->getType(), [Order::TYPE_DEVIS, Order::TYPE_TICKET])) {
            return;
        }

        /** @var Setting|null $minimumManagementFeesSetting */
        $minimumManagementFeesSetting = $this->repositorySetting->findOneBy([
            'code' => Setting::MINIMUM_MANAGEMENT_FEES,
            'active' => true,
        ]);

        /** @var Setting|null $managementFeesSetting */
        $managementFeesSetting = $this->repositorySetting->findOneBy([
            'code' => Setting::MANAGEMENT_FEES,
            'active' => true,
        ]);

        $order->removeAdjustments(Adjustment::MANAGEMENT_FEES_ADJUSTMENT);

        if (
            $minimumManagementFeesSetting
            && $managementFeesSetting
            && $order->getItemsTotal() <= (int) $minimumManagementFeesSetting->getContent()
        ) {
            /** @var Adjustment $adjustment */
            $adjustment = $this->factoryAdjustment->createNew();
            $adjustment->setType(Adjustment::MANAGEMENT_FEES_ADJUSTMENT);
            $adjustment->setAmount((int) $managementFeesSetting->getContent());

            $order->addAdjustment($adjustment);
        }
    }
}
