<?php

declare(strict_types=1);

namespace App\Repository;

use App\Entity\Customer;
use App\Entity\Order;
use Doctrine\ORM\NonUniqueResultException;
use Sylius\Bundle\CoreBundle\Doctrine\ORM\OrderRepository as BaseOrderRepository;
use Sylius\Component\Core\Model\CustomerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Core\OrderPaymentStates;

class OrderRepository extends BaseOrderRepository
{
    /**
     * @param string $tokenValue
     *
     * @return Order|null
     *
     * @throws NonUniqueResultException
     */
    public function findActiveByOrderToken(string $tokenValue): ?Order
    {
        return $this->createQueryBuilder('o')
            ->andWhere('o.orderToken = :tokenValue')
            ->andWhere('o.state IN (:states)')
            ->setParameters([
                'tokenValue' => $tokenValue,
                'states' => [Order::STATE_FULFILLED, Order::STATE_NEW],
            ])
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @param string $tokenValue
     *
     * @return Order|null
     *
     * @throws NonUniqueResultException
     */
    public function findOfferedByOrderToken(string $tokenValue): ?Order
    {
        return $this->createQueryBuilder('o')
            ->andWhere('o.orderToken = :tokenValue')
            ->andWhere('o.state = :state')
            ->setParameters([
                'tokenValue' => $tokenValue,
                'state' => Order::STATE_OFFERED,
            ])
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @param Customer $customer
     *
     * @return Order|null
     *
     * @throws NonUniqueResultException
     */
    public function findLastOrderByCustomer(Customer $customer): ?Order
    {
        return $this->createQueryBuilder('o')
            ->innerJoin('o.customer', 'c')
            ->andWhere('c.id = :id')
            ->andWhere('o.state = :state')
            ->setParameter('id', $customer->getId())
            ->setParameter('state', Order::STATE_FULFILLED)
            ->addOrderBy('o.id', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @param Customer $customer
     *
     * @return Order|null
     *
     * @throws NonUniqueResultException
     */
    public function findLastOrderNewFulfilledByCustomer(Customer $customer): ?Order
    {
        return $this->createQueryBuilder('o')
            ->innerJoin('o.customer', 'c')
            ->andWhere('c.id = :id')
            ->andWhere('o.state IN (:states)')
            ->setParameter('id', $customer->getId())
            ->setParameter('states', [Order::STATE_NEW, Order::STATE_FULFILLED])
            ->addOrderBy('o.id', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @param ProductInterface  $product
     * @param CustomerInterface $customer
     *
     * @return bool
     *
     * @throws NonUniqueResultException
     */
    public function alreadyOrderProduct(ProductInterface $product, CustomerInterface $customer): bool
    {
        return 0 < (int) $this->createQueryBuilder('o')
            ->select('COUNT(o.id)')
            ->innerJoin('o.items', 'i')
            ->innerJoin('i.variant', 'v')
            ->innerJoin('v.product', 'p')
            ->andWhere('o.customer = :customer')
            ->andWhere('o.state IN (:orders)')
            ->andWhere('o.paymentState = :payment')
            ->andWhere('p = :product')
            ->setParameter('product', $product)
            ->setParameter('customer', $customer)
            ->setParameter('orders', [OrderInterface::STATE_NEW, OrderInterface::STATE_FULFILLED])
            ->setParameter('payment', OrderPaymentStates::STATE_PAID)
            ->getQuery()
            ->getSingleScalarResult();
    }
}
