<?php

declare(strict_types=1);

namespace App\Service;

use App\Entity\Customer;
use Doctrine\ORM\EntityManagerInterface;

class IdentifierService
{
    private const IDENTIFIER_LENGTH = 17;
    private $entityManager;

    public function __construct(EntityManagerInterface $entityManager)
    {
        $this->entityManager = $entityManager;
    }

    public function createIdentifier(Customer $customer): ?string
    {
        if (empty($customer->getLastName()) && empty($customer->getFirstName()) && empty($customer->getCompanyName())) {
            return null;
        }

        if (Customer::TYPE_PARTICULIER === $customer->getType()) {
            $identifier = $customer->getLastName().$customer->getFirstName();
        } else {
            $identifier = $customer->getCompanyName();
        }

        $identifierAdjustment = null;

        do {
            $identifier = $this->transform($identifier, $identifierAdjustment);
            $customerRetrieved = $this->entityManager->getRepository(Customer::class)->findOneBy(['identifier' => $identifier]);

            if (is_null($identifierAdjustment)) {
                $identifierAdjustment = 0;
            } else {
                ++$identifierAdjustment;
            }
        } while ($customerRetrieved);

        return $identifier;
    }

    private function transform(string $text, int $adjustment = null): ?string
    {
        $text = trim($text);
        $text = str_replace([' ', '-', '°'], '', $text);
        $text = substr($text, 0, self::IDENTIFIER_LENGTH);
        $text = $this->stripAccents($text);
        $text = strtoupper($text);
        $text = trim($text);

        if (self::IDENTIFIER_LENGTH === strlen($text)) {
            $text = substr($text, 0, self::IDENTIFIER_LENGTH - strlen((string) $adjustment));
        }

        if (!is_null($adjustment)) {
            $text .= $adjustment;
        }

        return $text;
    }

    private function stripAccents(string $str): string
    {
        return strtr(utf8_decode($str), utf8_decode('àáâãäçèéêëìíîïñòóôõöùúûüýÿÀÁÂÃÄÇÈÉÊËÌÍÎÏÑÒÓÔÕÖÙÚÛÜÝ'), 'aaaaaceeeeiiiinooooouuuuyyAAAAACEEEEIIIINOOOOOUUUUY');
    }
}
