<?php

declare(strict_types=1);

namespace App\Service;

use App\Entity\Customer;
use App\Entity\Order;
use App\Model\AdminOrderCart;
use App\Model\AdminOrderInterface;
use App\Model\AdminOrderQuotation;
use App\Model\AdminOrderTicket;
use Doctrine\DBAL\ConnectionException;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\LoggerInterface;
use SM\Factory\Factory;
use Sylius\Component\Core\Model\Address;
use Sylius\Component\Core\Model\AddressInterface;
use Sylius\Component\Core\Model\AdminUserInterface;
use Sylius\Component\Core\Model\ChannelPricing;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\OrderItem;
use Sylius\Component\Core\Model\OrderItemUnit;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Core\Model\ProductVariant;
use Sylius\Component\Mailer\Sender\Sender;
use Sylius\Component\Order\Modifier\OrderItemQuantityModifierInterface;
use Sylius\Component\Order\Modifier\OrderModifierInterface;
use Sylius\Component\Order\OrderTransitions;
use Sylius\Component\Order\Processor\OrderProcessorInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Generator\RandomnessGeneratorInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Symfony\Component\Routing\RouterInterface;

class OrderService
{
    /** @var EntityManagerInterface */
    protected $entityManager;

    /** @var FactoryInterface */
    protected $factoryAddress;

    /** @var FactoryInterface */
    protected $factoryCustomer;

    /** @var FactoryInterface */
    protected $factoryOrder;

    /** @var FactoryInterface */
    protected $factoryPayment;

    /** @var FactoryInterface */
    protected $factoryShipment;

    /** @var FactoryInterface */
    protected $factoryShopUser;

    /** @var RepositoryInterface */
    protected $repositoryChannel;

    /** @var RepositoryInterface */
    protected $repositoryCustomer;

    /** @var RepositoryInterface */
    protected $repositoryPaymentMethod;

    /** @var RepositoryInterface */
    protected $repositoryProduct;

    /** @var RepositoryInterface */
    protected $repositoryShipmentMethod;

    /** @var Factory */
    protected $stateMachineFactory;

    /** @var Sender */
    protected $emailSender;

    /** @var RouterInterface */
    protected $router;

    /** @var LoggerInterface */
    protected $logger;

    /** @var string */
    protected $kernelProjectDir;

    /** @var OrderModifierInterface */
    protected $orderModifier;

    /** @var OrderItemQuantityModifierInterface */
    protected $orderItemQuantityModifier;

    /** @var FactoryInterface */
    protected $orderItemFactory;

    /** @var OrderProcessorInterface */
    protected $orderProcessor;

    /** @var EntityManagerInterface */
    protected $orderManager;

    /** @var RandomnessGeneratorInterface */
    protected $generator;

    /** @var IdentifierService */
    protected $identifierService;

    /**
     * OrderService constructor.
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        FactoryInterface $factoryAddress,
        FactoryInterface $factoryCustomer,
        FactoryInterface $factoryOrder,
        FactoryInterface $factoryPayment,
        FactoryInterface $factoryShipment,
        FactoryInterface $factoryShopUser,
        RepositoryInterface $repositoryChannel,
        RepositoryInterface $repositoryCustomer,
        RepositoryInterface $repositoryPaymentMethod,
        RepositoryInterface $repositoryProduct,
        RepositoryInterface $repositoryShipmentMethod,
        Factory $stateMachineFactory,
        Sender $emailSender,
        RouterInterface $router,
        LoggerInterface $logger,
        string $kernelProjectDir,
        OrderModifierInterface $orderModifier,
        OrderItemQuantityModifierInterface $orderItemQuantityModifier,
        FactoryInterface $orderItemFactory,
        OrderProcessorInterface $orderProcessor,
        EntityManagerInterface $orderManager,
        RandomnessGeneratorInterface $generator,
        IdentifierService $identifierService
    ) {
        $this->entityManager = $entityManager;
        $this->factoryAddress = $factoryAddress;
        $this->factoryCustomer = $factoryCustomer;
        $this->factoryOrder = $factoryOrder;
        $this->factoryPayment = $factoryPayment;
        $this->factoryShipment = $factoryShipment;
        $this->factoryShopUser = $factoryShopUser;
        $this->repositoryChannel = $repositoryChannel;
        $this->repositoryCustomer = $repositoryCustomer;
        $this->repositoryPaymentMethod = $repositoryPaymentMethod;
        $this->repositoryProduct = $repositoryProduct;
        $this->repositoryShipmentMethod = $repositoryShipmentMethod;
        $this->stateMachineFactory = $stateMachineFactory;
        $this->emailSender = $emailSender;
        $this->router = $router;
        $this->logger = $logger;
        $this->kernelProjectDir = $kernelProjectDir;
        $this->orderModifier = $orderModifier;
        $this->orderItemQuantityModifier = $orderItemQuantityModifier;
        $this->orderItemFactory = $orderItemFactory;
        $this->orderProcessor = $orderProcessor;
        $this->orderManager = $orderManager;
        $this->generator = $generator;
        $this->identifierService = $identifierService;
    }

    /**
     * @throws ConnectionException
     */
    public function createCart(OrderInterface $order, AdminUserInterface $salesman): OrderInterface
    {
        $connection = $this->entityManager->getConnection();

        try {
            $connection->beginTransaction();

            $order = $this->createUnknownAddress(new AdminOrderCart(), $order);
            $order->setOrderToken($this->generator->generateUriSafeString(10));
            $order->setSalesman($salesman);
            $order->removeShipments();
            $order->removePayments();
            $order = $this->processOrder($order);

            $this->entityManager->flush();

            $connection->commit();
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            $connection->rollBack();

            throw $e;
        }

        $this->emailSender->send('order_cart', [$order->getCustomer()->getEmail()], [
            'link' => $this->router->generate(
                'app_cart',
                [
                    'tokenValue' => $order->getOrderToken(),
                    'from' => 'email',
                ],
                RouterInterface::ABSOLUTE_URL
            ),
            'order' => $order,
        ]);

        return $order;
    }

    /**
     * @throws ConnectionException
     */
    public function createTicket(AdminOrderTicket $adminOrderTicket, AdminUserInterface $salesman): Order
    {
        $connection = $this->entityManager->getConnection();

        try {
            $connection->beginTransaction();
            $product = $this->repositoryProduct->findOneBy(['code' => 'TICKET']);

            if (!$product) {
                throw new \Exception();
            }

            $order = $this->createOrder($adminOrderTicket, Order::TYPE_TICKET, $product);
            $order->setOrderToken($this->generator->generateUriSafeString(10));
            $order->setSalesman($salesman);
            $order->removeShipments();
            $order->removePayments();

            /** @var OrderItem $item */
            foreach ($order->getItems() as $item) {
                /** @var OrderItemUnit $unit */
                foreach ($item->getUnits() as $unit) {
                    $unit->setShipment(null);
                }
            }

            $this->entityManager->persist($order);
            $this->entityManager->flush();
            $connection->commit();
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            $connection->rollBack();

            throw $e;
        }

        $this->emailSender->send('order_ticket', [$order->getCustomer()->getEmail()], [
            'link' => $this->router->generate(
                'app_cart',
                [
                    'tokenValue' => $order->getOrderToken(),
                    'from' => 'email',
                ],
                RouterInterface::ABSOLUTE_URL
            ),
            'order' => $order,
        ]);

        return $order;
    }

    /**
     * @throws ConnectionException
     */
    public function createCartFromOrder(OrderInterface $cart, OrderInterface $order): OrderInterface
    {
        $connection = $this->entityManager->getConnection();

        try {
            $connection->beginTransaction();

            $cart->setQuotationPdf($order->getQuotationPdf());
            $cart->setType($order->getType());
            $cart->setChannel($order->getChannel());
            $cart->setCurrencyCode($order->getCurrencyCode());
            $cart->setLocaleCode($order->getLocaleCode());
            $cart->setCreatedAt($order->getCreatedAt());
            $cart->setUpdatedAt($order->getUpdatedAt());
            $cart->setSalesman($order->getSalesman());
            $cart->setOrderToken($order->getOrderToken());

            /* @var OrderItemUnitInterface $item */
            foreach ($order->getItemUnits() as $orderItemUnit) {
                $orderItem = $orderItemUnit->getOrderItem();
                $cart = $this->addProductToOrder($cart, $orderItem->getVariant()->getProduct(), 1, $orderItem->getUnitPrice());
            }

            $this->processOrder($cart);
            $connection->commit();

            return $cart;
        } catch (\Exception $e) {
            $connection->rollBack();

            throw $e;
        }
    }

    /**
     * @throws ConnectionException
     */
    public function createQuotation(AdminOrderQuotation $adminOrderQuotation, AdminUserInterface $salesman): Order
    {
        $connection = $this->entityManager->getConnection();

        try {
            $file = $adminOrderQuotation->getQuotation();
            $connection->beginTransaction();

            $fileName = md5(uniqid('', true)).'.'.$file->guessExtension();
            $file->move(
                $this->getQuotationDir(),
                $fileName
            );

            $product = $this->repositoryProduct->findOneBy(['code' => 'QUOTATION']);

            if (!$product) {
                throw new \Exception();
            }

            $order = $this->createOrder($adminOrderQuotation, Order::TYPE_DEVIS, $product);
            $order->setOrderToken($this->generator->generateUriSafeString(10));
            $order->setQuotationPdf($fileName);
            $order->setSalesman($salesman);
            $order->removeShipments();
            $order->removePayments();

            /** @var OrderItem $item */
            foreach ($order->getItems() as $item) {
                /** @var OrderItemUnit $unit */
                foreach ($item->getUnits() as $unit) {
                    $unit->setShipment(null);
                }
            }

            $this->entityManager->persist($order);
            $this->entityManager->flush();
            $connection->commit();
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            $connection->rollBack();

            throw $e;
        }

        $this->emailSender->send('order_quotation', [$order->getCustomer()->getEmail()], [
            'link' => $this->router->generate(
                'app_cart',
                [
                    'tokenValue' => $order->getOrderToken(),
                    'from' => 'email',
                ],
                RouterInterface::ABSOLUTE_URL
            ),
        ]);

        return $order;
    }

    public function getQuotationDir(): string
    {
        return $this->kernelProjectDir.'/data/quotation';
    }

    /**
     * @throws \Exception
     */
    public function addProductToOrder(
        OrderInterface $order,
        ProductInterface $product,
        int $quantity = 1,
        int $unitPrice = null
    ): OrderInterface {
        if (0 === $product->getVariants()->count()) {
            throw new \Exception("Product {$product->getCode()} has not variant.");
        }

        /** @var ProductVariant $productVariant */
        $productVariant = $product->getVariants()->first();

        /** @var ChannelPricing $channelPricing */
        $channelPricing = $productVariant->getChannelPricings()->first();

        /* Update the price */
        if (!is_null($unitPrice)) {
            $channelPricing->setOriginalPrice($unitPrice);
            $channelPricing->setPrice($unitPrice);
        }

        /** @var OrderItem $orderItem */
        $orderItem = $this->orderItemFactory->createNew();
        $orderItem->setVariant($productVariant);

        $this->orderItemQuantityModifier->modify($orderItem, $quantity);
        $this->orderModifier->addToOrder($order, $orderItem);

        return $order;
    }

    public function processOrder(OrderInterface $order): OrderInterface
    {
        $this->orderProcessor->process($order);
        $this->orderManager->persist($order);
        $this->orderManager->flush();

        return $order;
    }

    public function removeOrder(OrderInterface $order): void
    {
        $this->orderManager->remove($order);
        $this->orderManager->flush();
    }

    /**
     * @throws \Exception
     */
    protected function createOrder(AdminOrderInterface $adminOrder, int $type, ProductInterface $product): Order
    {
        $createdAt = new \DateTime();
        $updatedAt = new \DateTime();

        $price = (float) str_replace(',', '.', $adminOrder->getTotal());
        // @todo use taxzone
        $price = $price / 1.2 * 100;
        $price = (int) round($price);

        /** @var Order $order */
        $order = $this->factoryOrder->createNew();
        $order->setType($type);
        $order->setChannel($this->repositoryChannel->findOneBy(['code' => 'ECOMMERCE']));
        $order->setCheckoutCompletedAt($createdAt);
        $order->setCurrencyCode('EUR');
        $order->setLocaleCode('fr_FR');
        $order->setCreatedAt($createdAt);
        $order->setUpdatedAt($updatedAt);

        $order = $this->addProductToOrder($order, $product, 1, $price);
        $customer = $this->repositoryCustomer->findOneBy(['email' => $adminOrder->getEmail()]);

        if (!$customer) {
            /** @var Customer $customer */
            $customer = $this->factoryCustomer->createNew();
            $customer->setCompanyName($adminOrder->getCompanyName() ?? 'unknown');
            $customer->setLastName($adminOrder->getLastName() ?? 'unknown');
            $customer->setFirstName($adminOrder->getFirstName() ?? 'unknown');
            $customer->setEmail($adminOrder->getEmail());
            $customer->setPhoneNumber($adminOrder->getPhone());
            $customer->setCreatedAt($createdAt);
            $customer->setUpdatedAt($updatedAt);
        }

        $order->setCustomer($customer);

        return $this->createUnknownAddress($adminOrder, $order);
    }

    /**
     * @throws \SM\SMException
     */
    public function createUnknownAddress(AdminOrderInterface $adminOrder, OrderInterface $order): OrderInterface
    {
        $createdAt = new \DateTime();
        $updatedAt = new \DateTime();

        /** @var Address $billingAddress */
        $billingAddress = $this->factoryAddress->createNew();
        $billingAddress->setLastName($adminOrder->getLastName() ?? 'unknown');
        $billingAddress->setFirstName($adminOrder->getFirstName() ?? 'unknown');
        $billingAddress->setCompany($adminOrder->getCompanyName() ?? 'unknown');
        $billingAddress->setPhoneNumber($adminOrder->getPhone());
        $billingAddress->setCreatedAt($createdAt);
        $billingAddress->setUpdatedAt($updatedAt);
        $billingAddress->setStreet('unknown');
        $billingAddress->setCity('unknown');
        $billingAddress->setPostcode('unknown');
        $billingAddress->setCountryCode('FR');
        $order->setBillingAddress($billingAddress);

        /** @var Address $shippingAddress */
        $shippingAddress = $this->factoryAddress->createNew();
        $shippingAddress->setLastName($adminOrder->getLastName() ?? 'unknown');
        $shippingAddress->setFirstName($adminOrder->getFirstName() ?? 'unknown');
        $shippingAddress->setCompany($adminOrder->getCompanyName() ?? 'unknown');
        $shippingAddress->setPhoneNumber($adminOrder->getPhone());
        $shippingAddress->setCreatedAt($createdAt);
        $shippingAddress->setUpdatedAt($updatedAt);
        $shippingAddress->setStreet('unknown');
        $shippingAddress->setCity('unknown');
        $shippingAddress->setPostcode('unknown');
        $shippingAddress->setCountryCode('FR');
        $order->setShippingAddress($shippingAddress);

        $stateMachine = $this->stateMachineFactory->get($order, OrderTransitions::GRAPH);

        $stateMachine->apply(OrderTransitions::TRANSITION_CREATE);
        $stateMachine->apply('offer');

        return $order;
    }

    public function setCustomerInformation(OrderInterface $order): void
    {
        /* @var AddressInterface $address */
        $address = $order->getShippingAddress();
        /* @var Customer $customer */
        $customer = $order->getCustomer();
        $customer->setLastName($address->getLastName());
        $customer->setFirstName($address->getFirstName());
        $customer->setCompanyName($address->getCompany());
        $customer->setType($address->getType());

        $this->entityManager->flush();
    }

    public function createIdentifier(OrderInterface $order): void
    {
        /* @var Customer $customer */
        $customer = $order->getCustomer();

        if ($customer->getIdentifier()) {
            return;
        }

        $customer->setIdentifier($this->identifierService->createIdentifier($customer));
        $this->entityManager->flush();
    }
}
