<?php

/*
 * This file is part of the Sylius package.
 *
 * (c) Paweł Jędrzejewski
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace Sylius\Bundle\ThemeBundle\Configuration\Filesystem;

use Sylius\Bundle\ThemeBundle\Configuration\ConfigurationSourceFactoryInterface;
use Sylius\Bundle\ThemeBundle\Locator\RecursiveFileLocator;
use Symfony\Component\Config\Definition\Builder\ArrayNodeDefinition;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Reference;

final class FilesystemConfigurationSourceFactory implements ConfigurationSourceFactoryInterface
{
    /**
     * {@inheritdoc}
     */
    public function buildConfiguration(ArrayNodeDefinition $node): void
    {
        $filesystemNode = $node->fixXmlConfig('directory', 'directories')->children();

        $filesystemNode
            ->scalarNode('filename')
                ->defaultValue('composer.json')
                ->cannotBeEmpty()
        ;

        $filesystemNode
            ->integerNode('scan_depth')
                ->info('Restrict depth to scan for configuration file inside theme folder')
                ->defaultNull()
        ;

        $filesystemNode
            ->arrayNode('directories')
                ->defaultValue(['%kernel.project_dir%/app/themes'])
                ->requiresAtLeastOneElement()
                ->performNoDeepMerging()
                ->prototype('scalar')
        ;
    }

    /**
     * {@inheritdoc}
     */
    public function initializeSource(ContainerBuilder $container, array $config): Definition
    {
        $recursiveFileLocator = new Definition(RecursiveFileLocator::class, [
            new Reference('sylius.theme.finder_factory'),
            $config['directories'],
            $config['scan_depth'],
        ]);

        $configurationLoader = new Definition(ProcessingConfigurationLoader::class, [
            new Definition(JsonFileConfigurationLoader::class, [
                new Reference('sylius.theme.filesystem'),
            ]),
            new Reference('sylius.theme.configuration.processor'),
        ]);

        $configurationProvider = new Definition(FilesystemConfigurationProvider::class, [
            $recursiveFileLocator,
            $configurationLoader,
            $config['filename'],
        ]);

        return $configurationProvider;
    }

    /**
     * {@inheritdoc}
     */
    public function getName(): string
    {
        return 'filesystem';
    }
}
